@extends('layouts.app')

@section('content')
<div class="container">
    <div class="card">
        <div class="card-header mb-5">
            <div class="flex justify-between items-center mb-8">
                <div>
                    <h1 class="text-3xl font-bold text-gray-800">{{ $report->report_name }}</h1>
                    <p class="text-gray-600 mt-2">
                        <strong>Phase:</strong> {{ $report->phase }} | 
                        <strong>Created:</strong> {{ $report->created_at->format('F d, Y H:i') }} | 
                        <strong>By:</strong> {{ $report->creator ? $report->creator->name : 'N/A' }}
                    </p>
                </div>
                <div>
                    <a href="{{ route('gpv.saved-reports') }}" class="bg-gray-500 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded inline-flex items-center">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Reports
                    </a>
                </div>
            </div>

            <!-- Report Metadata -->
            <div class="bg-gray-100 p-4 rounded-lg grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                <div>
                    <p class="text-sm text-gray-600">School</p>
                    <p class="font-semibold">{{ $report->school ? $report->school->name : 'All Schools' }}</p>
                </div>
                <div>
                    <p class="text-sm text-gray-600">Course</p>
                    <p class="font-semibold">{{ $report->course ? $report->course->name : 'All Courses' }}</p>
                </div>
                {{-- <div>
                    <p class="text-sm text-gray-600">Intake</p>
                    <p class="font-semibold">{{ $report->intake ? $report->intake->year : 'All Intakes' }}</p>
                </div> --}}
                <div>
                    <p class="text-sm text-gray-600">Total Cadets</p>
                    <p class="font-semibold text-blue-600" id="totalStudents">{{ $report->total_students }}</p>
                </div>
                @if($report->bn)
                <div>
                    <p class="text-sm text-gray-600">Bn</p>
                    <p class="font-semibold">{{ $report->bn }}</p>
                </div>
                @endif
                @if($report->coy)
                <div>
                    <p class="text-sm text-gray-600">Coy</p>
                    <p class="font-semibold">{{ $report->coy }}</p>
                </div>
                @endif
                @if($report->pl)
                <div>
                    <p class="text-sm text-gray-600">Pl</p>
                    <p class="font-semibold">{{ $report->pl }}</p>
                </div>
                @endif
                @if($report->grade)
                <div>
                    <p class="text-sm text-gray-600">Grade Filter</p>
                    <p class="font-semibold">{{ $report->grade }}</p>
                </div>
                @endif
            </div>

            <!-- Search Box -->
            <div class="bg-white p-4 rounded-lg border border-gray-300 mb-4">
                <div class="flex items-center gap-4">
                    <div class="flex-grow">
                        <label for="searchInput" class="block text-sm font-medium text-gray-700 mb-2">Search by Army Number, Name:</label>
                        <input type="text" 
                               id="searchInput" 
                               placeholder="Type to search..." 
                               class="w-full px-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                               onkeyup="searchTable()">
                    </div>
                    <div class="flex items-end">
                        <button onclick="clearSearch()" class="bg-gray-400 hover:bg-gray-600 text-white font-bold py-2 px-4 rounded inline-flex items-center">
                            <i class="fas fa-times mr-2"></i> Clear
                        </button>
                    </div>
                </div>
                <p class="text-sm text-gray-500 mt-2">
                    Showing <span id="visibleCount" class="font-semibold text-blue-600">{{ $report->total_students }}</span> of {{ $report->total_students }} students
                </p>
            </div>
        </div>

        <!-- Export Buttons -->
        <div class="flex justify-end mb-4 space-x-4 px-4">
        
            <button onclick="exportToExcel()" class="bg-green-500 hover:bg-green-700 text-white font-bold py-2 px-4 rounded inline-flex items-center">
                <i class="fas fa-file-excel mr-2"></i> Export Excel
            </button>
        </div>

        <div class="card-body">
            <div class="overflow-x-auto">
                <table class="w-full table-auto" id="reportTable">
                    <thead>
                        <tr class="bg-gray-200 text-gray-700">
                            <th class="px-4 py-2">#</th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(1)">
                                Admission No. <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(2)">
                                Army Number <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(3)">
                                Rank <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(4)">
                                O/CDT Name <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(5)">
                                Theory <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(6)">
                                Practical <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(7)">
                                22CR <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(8)">
                                PR <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(9)">
                                Total <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(10)">
                                Grade <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(11)">
                                Comment <span class="sort-icon">↕️</span>
                            </th>
                            <th class="px-4 py-2 cursor-pointer" onclick="sortTable(12)">
                                Remarks <span class="sort-icon">↕️</span>
                            </th>
                        </tr>
                    </thead>
                    <tbody id="tableBody">
                        @foreach($report->details as $index => $detail)
                            <tr class="border-b hover:bg-gray-50 data-row {{ $index % 2 == 1 ? 'bg-gray-100' : '' }}" 
                                data-army="{{ strtolower($detail->army_number) }}"
                                data-name="{{ strtolower($detail->name) }}"
                                data-rank="{{ strtolower($detail->rank) }}"
                                data-admission="{{ strtolower($detail->admission_number) }}">
                                <td class="px-4 py-2 row-number">{{ $index + 1 }}</td>
                                <td class="px-4 py-2">{{ $detail->admission_number }}</td>
                                <td class="px-4 py-2 font-semibold text-blue-600">{{ $detail->army_number }}</td>
                                <td class="px-4 py-2">{{ $detail->rank }}</td>
                                <td class="px-4 py-2">{{ $detail->name }}</td>
                                <td class="px-4 py-2">{{ number_format($detail->theory_score, 2) }}</td>
                                <td class="px-4 py-2">{{ number_format($detail->practical_score, 2) }}</td>
                                <td class="px-4 py-2">{{ number_format($detail->cr_22_score, 2) }}</td>
                                <td class="px-4 py-2">{{ number_format($detail->pr_score, 2) }}</td>
                                <td class="px-4 py-2 font-semibold">{{ number_format($detail->total_score, 2) }}</td>
                                <td class="px-4 py-2">
                                    <span class="px-2 py-1 rounded font-semibold
                                        @if($detail->grade == 'A') bg-green-100 text-green-800
                                        @elseif($detail->grade == 'B') bg-blue-100 text-blue-800
                                        @elseif($detail->grade == 'C') bg-yellow-100 text-yellow-800
                                        @elseif($detail->grade == 'D') bg-orange-100 text-orange-800
                                        @else bg-red-100 text-red-800
                                        @endif">
                                        {{ $detail->grade }}
                                    </span>
                                </td>
                                <td class="px-4 py-2">{{ $detail->comment }}</td>
                                <td class="px-4 py-2">
                                    <span class="px-2 py-1 rounded font-semibold {{ $detail->remarks == 'PASSED' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                                        {{ $detail->remarks }}
                                    </span>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- No Results Message -->
            <div id="noResults" class="text-center py-8 text-gray-500" style="display: none;">
                <i class="fas fa-search text-4xl mb-2"></i>
                <p class="text-lg">No students found matching your search</p>
                <button onclick="clearSearch()" class="mt-2 text-blue-500 hover:underline">Clear search</button>
            </div>
        </div>
    </div>
</div>

<style>
    @media print {
        .flex, .bg-gray-500, .bg-green-500, .bg-white.p-4.rounded-lg.border {
            display: none !important;
        }
        .sort-icon {
            display: none !important;
        }
    }

    .data-row {
        transition: all 0.3s ease;
    }

    .highlight {
        background-color: #fef3c7 !important;
        animation: highlight-fade 2s ease-in-out;
    }

    @keyframes highlight-fade {
        0% { background-color: #fef3c7; }
        100% { background-color: inherit; }
    }
</style>

<script>
let sortDirection = Array(13).fill('asc');
const totalStudents = {{ $report->total_students }};

// Search function
function searchTable() {
    const input = document.getElementById('searchInput');
    const filter = input.value.toLowerCase().trim();
    const table = document.getElementById('tableBody');
    const rows = table.getElementsByClassName('data-row');
    const noResults = document.getElementById('noResults');
    let visibleCount = 0;

    for (let i = 0; i < rows.length; i++) {
        const armyNumber = rows[i].getAttribute('data-army');
        const name = rows[i].getAttribute('data-name');
        const rank = rows[i].getAttribute('data-rank');
        const admission = rows[i].getAttribute('data-admission');

        if (armyNumber.includes(filter) || 
            name.includes(filter) || 
            rank.includes(filter) ||
            admission.includes(filter)) {
            rows[i].style.display = '';
            visibleCount++;
            
            // Add highlight effect
            if (filter !== '') {
                rows[i].classList.add('highlight');
                setTimeout(() => {
                    rows[i].classList.remove('highlight');
                }, 2000);
            }
        } else {
            rows[i].style.display = 'none';
        }
    }

    // Update row numbers for visible rows
    updateRowNumbers();

    // Update visible count
    document.getElementById('visibleCount').textContent = visibleCount;

    // Show/hide no results message
    if (visibleCount === 0 && filter !== '') {
        noResults.style.display = 'block';
        table.parentElement.style.display = 'none';
    } else {
        noResults.style.display = 'none';
        table.parentElement.style.display = 'block';
    }
}

// Clear search
function clearSearch() {
    document.getElementById('searchInput').value = '';
    searchTable();
}

// Update row numbers for visible rows
function updateRowNumbers() {
    const table = document.getElementById('tableBody');
    const rows = table.getElementsByClassName('data-row');
    let visibleIndex = 1;

    for (let i = 0; i < rows.length; i++) {
        if (rows[i].style.display !== 'none') {
            rows[i].querySelector('.row-number').textContent = visibleIndex;
            visibleIndex++;
        }
    }
}

// Sort table
function sortTable(columnIndex) {
    const table = document.getElementById('tableBody');
    const rows = Array.from(table.getElementsByClassName('data-row'));
    
    sortDirection[columnIndex] = sortDirection[columnIndex] === 'asc' ? 'desc' : 'asc';
    
    // Update sort icons
    document.querySelectorAll('.sort-icon').forEach(icon => {
        icon.textContent = '↕️';
    });
    
    const sortIcon = document.querySelectorAll('.sort-icon')[columnIndex - 1];
    sortIcon.textContent = sortDirection[columnIndex] === 'asc' ? '↑' : '↓';
    
    // Sort rows
    rows.sort((a, b) => {
        let aValue = a.cells[columnIndex].textContent.trim();
        let bValue = b.cells[columnIndex].textContent.trim();
        
        // Check if values are numbers
        const aNum = parseFloat(aValue.replace(/,/g, ''));
        const bNum = parseFloat(bValue.replace(/,/g, ''));
        
        if (!isNaN(aNum) && !isNaN(bNum)) {
            aValue = aNum;
            bValue = bNum;
        }
        
        if (aValue < bValue) {
            return sortDirection[columnIndex] === 'asc' ? -1 : 1;
        } else if (aValue > bValue) {
            return sortDirection[columnIndex] === 'asc' ? 1 : -1;
        }
        return 0;
    });
    
    // Re-append sorted rows
    rows.forEach(row => table.appendChild(row));
    
    // Reapply zebra striping
    rows.forEach((row, index) => {
        row.classList.remove('bg-gray-100');
        if (index % 2 === 1) {
            row.classList.add('bg-gray-100');
        }
    });

    // Update row numbers after sorting
    updateRowNumbers();
}

// Print report
function printReport() {
    // Hide search box before printing
    const searchBox = document.querySelector('.bg-white.p-4.rounded-lg.border');
    if (searchBox) searchBox.style.display = 'none';
    
    window.print();
    
    // Show search box after printing
    if (searchBox) searchBox.style.display = 'block';
}

// Export to Excel/CSV
function exportToExcel() {
    let csv = [];
    let table = document.getElementById('reportTable');
    
    // Add report header
    csv.push('"{{ $report->report_name }}"');
    csv.push('"Phase: {{ $report->phase }}"');
    csv.push('"School: {{ $report->school ? $report->school->name : 'All Schools' }}"');
    csv.push('"Course: {{ $report->course ? $report->course->name : 'All Courses' }}"');
    csv.push('""'); // Empty line
    
    // Get headers
    let headers = [];
    let headerCells = table.querySelectorAll('thead th');
    headerCells.forEach(cell => {
        headers.push('"' + cell.textContent.replace(/[↕️↑↓]/g, '').trim() + '"');
    });
    csv.push(headers.join(','));
    
    // Get only visible rows
    let visibleRows = Array.from(table.querySelectorAll('tbody tr')).filter(row => {
        return row.style.display !== 'none';
    });
    
    visibleRows.forEach(row => {
        let rowData = [];
        let cells = row.querySelectorAll('td');
        
        cells.forEach(cell => {
            let data = cell.textContent.replace(/(\r\n|\n|\r)/gm, '').replace(/(\s\s+)/gm, ' ').trim();
            data = data.replace(/"/g, '""');
            rowData.push('"' + data + '"');
        });
        
        csv.push(rowData.join(','));
    });
    
    // Create and download file
    let csvFile = new Blob([csv.join("\n")], { type: "text/csv;charset=utf-8;" });
    let downloadLink = document.createElement("a");
    downloadLink.download = "{{ $report->report_name }}_" + new Date().toISOString().slice(0,10) + ".csv";
    downloadLink.href = window.URL.createObjectURL(csvFile);
    downloadLink.style.display = "none";
    document.body.appendChild(downloadLink);
    downloadLink.click();
    document.body.removeChild(downloadLink);
}

// Add keyboard shortcut for search (Ctrl+F or Cmd+F)
document.addEventListener('keydown', function(e) {
    if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
        e.preventDefault();
        document.getElementById('searchInput').focus();
    }
});
</script>
@endsection