<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\RemarkTemplate;
use App\Models\Phase;

class RemarkTemplateSeeder extends Seeder
{
    public function run(): void
    {
        $phases = Phase::all();

        if ($phases->isEmpty()) {
            $this->command->warn('⚠️  No phases found. Please seed phases first.');
            return;
        }

        $this->command->info("Found {$phases->count()} phase(s). Seeding remarks...");

        foreach ($phases as $phase) {
            $count = $this->seedForPhase($phase);
            $this->command->info("✅ Phase [{$phase->name}] — {$count} templates saved.");
        }

        $this->command->info('Done. Total rows in remark_templates: ' . RemarkTemplate::count());
    }

    private function seedForPhase($phase): int
    {
        $count = 0;
        foreach ($this->getTemplates() as $tpl) {
            RemarkTemplate::updateOrCreate(
                [
                    'phase_id' => $phase->id,
                    'grade'    => $tpl['grade'],
                    'type'     => $tpl['type'],
                ],
                ['remark_text' => $tpl['remark_text']]
            );
            $count++;
        }
        return $count;
    }

    /**
     * 24 templates: 6 grades × 4 roles.
     * Placeholders replaced at render time:
     *   {{name}}        → student's name (uppercase)
     *   {{phase}}       → phase name e.g. "Phase 1"
     *   {{gender_he}}   → he / she
     *   {{gender_He}}   → He / She
     *   {{gender_his}}  → his / her
     *   {{gender_His}}  → His / Her
     *   {{gender_him}}  → him / her
     *   {{gender_Him}}  → Him / Her
     */
    private function getTemplates(): array
    {
        return [

            // ══════════════════════════════════════
            // GRADE A — EXCELLENT
            // ══════════════════════════════════════
            [
                'grade' => 'A', 'type' => 'PL_COMD',
                'remark_text' => 'O/Cdt {{name}} has demonstrated an outstanding level of performance throughout {{gender_his}} training in {{phase}}. {{gender_He}} consistently exhibited exemplary military bearing, leadership qualities, and academic excellence. I highly recommend {{gender_him}} for commissioning.',
            ],
            [
                'grade' => 'A', 'type' => 'OC',
                'remark_text' => 'O/Cdt {{name}} has excelled in all aspects of {{phase}} training. {{gender_His}} dedication, discipline, and intellectual capacity are commendable. {{gender_He}} has set a high standard for {{gender_his}} peers and is strongly recommended for further advancement.',
            ],
            [
                'grade' => 'A', 'type' => 'CI',
                'remark_text' => 'O/Cdt {{name}} has performed exceptionally well during {{phase}}. {{gender_He}} has demonstrated superior understanding of all subjects and exercises. {{gender_His}} conduct, attitude, and professional competence are of the highest standard. Strongly recommended.',
            ],
            [
                'grade' => 'A', 'type' => 'COMDT',
                'remark_text' => 'O/Cdt {{name}} has achieved an excellent result in {{phase}}. {{gender_He}} has displayed outstanding qualities befitting a commissioned officer. I am confident that {{gender_he}} will serve the UPDF with great distinction. Highly recommended for commissioning.',
            ],

            // ══════════════════════════════════════
            // GRADE B — VERY GOOD
            // ══════════════════════════════════════
            [
                'grade' => 'B', 'type' => 'PL_COMD',
                'remark_text' => 'O/Cdt {{name}} has performed very well during {{phase}}. {{gender_He}} has shown a strong grasp of military concepts and consistent commitment to {{gender_his}} duties. {{gender_His}} leadership potential is evident and {{gender_he}} is recommended for commissioning.',
            ],
            [
                'grade' => 'B', 'type' => 'OC',
                'remark_text' => 'O/Cdt {{name}} has shown very good performance throughout {{phase}}. {{gender_He}} is disciplined, motivated, and demonstrates a clear understanding of military requirements. {{gender_He}} is recommended for the next phase of training.',
            ],
            [
                'grade' => 'B', 'type' => 'CI',
                'remark_text' => 'O/Cdt {{name}} has achieved a very good standard in {{phase}}. {{gender_He}} has engaged positively in all training activities and shown good professional development. {{gender_His}} overall performance reflects well on {{gender_his}} potential as a future officer.',
            ],
            [
                'grade' => 'B', 'type' => 'COMDT',
                'remark_text' => 'O/Cdt {{name}} has performed very well in {{phase}}. {{gender_He}} has shown the qualities expected of a commissioned officer. I recommend {{gender_him}} for progression to the next phase of officer training.',
            ],

            // ══════════════════════════════════════
            // GRADE C — GOOD
            // ══════════════════════════════════════
            [
                'grade' => 'C', 'type' => 'PL_COMD',
                'remark_text' => 'O/Cdt {{name}} has achieved a good standard during {{phase}}. {{gender_He}} has met the required training objectives and shown satisfactory military conduct. With continued effort, {{gender_he}} has the potential to develop further as an officer.',
            ],
            [
                'grade' => 'C', 'type' => 'OC',
                'remark_text' => 'O/Cdt {{name}} has performed to a good standard in {{phase}}. {{gender_He}} has demonstrated adequate understanding of the training programme. I encourage {{gender_him}} to put in additional effort to improve {{gender_his}} overall performance.',
            ],
            [
                'grade' => 'C', 'type' => 'CI',
                'remark_text' => 'O/Cdt {{name}} has shown a good level of performance in {{phase}}. {{gender_He}} has adequately met the training standards set for this phase. {{gender_He}} is advised to work harder in areas where improvement is required.',
            ],
            [
                'grade' => 'C', 'type' => 'COMDT',
                'remark_text' => 'O/Cdt {{name}} has achieved a good result in {{phase}}. {{gender_He}} is progressing satisfactorily in {{gender_his}} training and is recommended to continue with additional focus on identified areas for improvement.',
            ],

            // ══════════════════════════════════════
            // GRADE D — SATISFACTORY
            // ══════════════════════════════════════
            [
                'grade' => 'D', 'type' => 'PL_COMD',
                'remark_text' => 'O/Cdt {{name}} has achieved a satisfactory standard in {{phase}}. {{gender_He}} has met the minimum requirements but needs to demonstrate greater commitment and effort. {{gender_He}} is advised to improve {{gender_his}} performance significantly in the next phase.',
            ],
            [
                'grade' => 'D', 'type' => 'OC',
                'remark_text' => 'O/Cdt {{name}} has performed satisfactorily in {{phase}}. While {{gender_he}} has met the basic training requirements, there are notable areas requiring improvement. {{gender_He}} is cautioned to apply more diligence going forward.',
            ],
            [
                'grade' => 'D', 'type' => 'CI',
                'remark_text' => 'O/Cdt {{name}} has achieved a satisfactory standard in {{phase}}. {{gender_His}} performance, while passing, requires significant improvement. {{gender_He}} is strongly advised to dedicate more effort to both theoretical and practical components of training.',
            ],
            [
                'grade' => 'D', 'type' => 'COMDT',
                'remark_text' => 'O/Cdt {{name}} has achieved a satisfactory result in {{phase}}. {{gender_He}} is advised to improve {{gender_his}} overall performance. Continued progress is required to meet the standards expected of a commissioned officer.',
            ],

            // ══════════════════════════════════════
            // GRADE E — FAIR
            // ══════════════════════════════════════
            [
                'grade' => 'E', 'type' => 'PL_COMD',
                'remark_text' => 'O/Cdt {{name}} has achieved a fair standard in {{phase}}. {{gender_His}} performance has been below expectations and requires immediate and sustained improvement. {{gender_He}} is strongly advised to re-examine {{gender_his}} commitment to the training programme.',
            ],
            [
                'grade' => 'E', 'type' => 'OC',
                'remark_text' => 'O/Cdt {{name}} has shown a fair level of performance during {{phase}}. {{gender_He}} has passed, but only marginally. A serious improvement in attitude, application, and results is required in the next phase.',
            ],
            [
                'grade' => 'E', 'type' => 'CI',
                'remark_text' => 'O/Cdt {{name}} has performed at a fair level in {{phase}}. {{gender_He}} narrowly met the passing standard. A substantial improvement in {{gender_his}} academic and practical performance is necessary if {{gender_he}} is to progress as an officer.',
            ],
            [
                'grade' => 'E', 'type' => 'COMDT',
                'remark_text' => 'O/Cdt {{name}} has achieved a fair result in {{phase}} and has only just met the minimum standard. {{gender_He}} is placed on notice to improve {{gender_his}} performance substantially. Failure to improve may affect {{gender_his}} continuation in training.',
            ],

            // ══════════════════════════════════════
            // GRADE F — FAILED
            // ══════════════════════════════════════
            [
                'grade' => 'F', 'type' => 'PL_COMD',
                'remark_text' => 'O/Cdt {{name}} has failed to meet the minimum required standard during {{phase}}. {{gender_His}} performance has been consistently below the acceptable threshold. It is recommended that {{gender_his}} case be reviewed in accordance with applicable training regulations.',
            ],
            [
                'grade' => 'F', 'type' => 'OC',
                'remark_text' => 'O/Cdt {{name}} has not met the required standard for {{phase}}. {{gender_His}} performance has been unsatisfactory across multiple components. I recommend that {{gender_his}} case be referred to higher authority for appropriate action.',
            ],
            [
                'grade' => 'F', 'type' => 'CI',
                'remark_text' => 'O/Cdt {{name}} has failed {{phase}}. Despite guidance and support, {{gender_his}} results have not met the minimum passing standard. A review of {{gender_his}} continued participation in the Officer Cadet Course is strongly advised.',
            ],
            [
                'grade' => 'F', 'type' => 'COMDT',
                'remark_text' => 'O/Cdt {{name}} has failed to achieve the required standard in {{phase}}. Having considered {{gender_his}} overall performance, I do not recommend {{gender_him}} for progression at this time. {{gender_His}} case is referred for review in accordance with UPDF training regulations.',
            ],
        ];
    }
}