<?php

namespace App\Services;

use App\Models\PhaseConfig;
use App\Models\GradeScale;

class PhaseConfigService
{
    /**
     * Get configuration for a specific phase
     *
     * @param string|null $phaseName - Phase name or null for active phase
     * @return array
     */
    public function getPhaseConfig(?string $phaseName = null): array
    {
        // If no phase name provided, get the active phase
        if ($phaseName === null) {
            $phaseConfig = PhaseConfig::where('is_active', true)->first();
        } else {
            $phaseConfig = PhaseConfig::where('phase_name', $phaseName)->first();
        }

        // If no config found, return empty array
        if (!$phaseConfig) {
            return [];
        }

        return [
            'msk_theory' => $phaseConfig->msk_theory_config,
            'msk_practical' => $phaseConfig->msk_practical_config,
            'character_22cr' => $phaseConfig->character_22cr_config,
            'character_pr' => $phaseConfig->character_pr_config
        ];
    }

    /**
     * Get passing criteria for a specific phase
     *
     * @param string|null $phaseName - Phase name or null for active phase
     * @return array
     */
    public function getPassingCriteria(?string $phaseName = null): array
    {
        // If no phase name provided, get the active phase
        if ($phaseName === null) {
            $phaseConfig = PhaseConfig::where('is_active', true)->first();
        } else {
            $phaseConfig = PhaseConfig::where('phase_name', $phaseName)->first();
        }

        // If no config found, return empty array
        if (!$phaseConfig) {
            return [];
        }

        return $phaseConfig->passing_criteria;
    }

    /**
     * Get max scores for a specific phase
     *
     * @param string|null $phaseName - Phase name or null for active phase
     * @return array
     */
    public function getMaxScores(?string $phaseName = null): array
    {
        // If no phase name provided, get the active phase
        if ($phaseName === null) {
            $phaseConfig = PhaseConfig::where('is_active', true)->first();
        } else {
            $phaseConfig = PhaseConfig::where('phase_name', $phaseName)->first();
        }

        // If no config found, return the default
        if (!$phaseConfig) {
            // You may want to define a default or throw an exception here
            return [
                'theory' => 40,
                'practical' => 30,
                '22cr' => 20,
                'pr' => 10
            ];
        }

        return $phaseConfig->max_scores;
    }

    /**
     * Get grade scale configuration
     *
     * @return array
     */
    public function getGradeScale(): array
    {
        $grades = GradeScale::orderBy('min_score', 'desc')->get();
        
        $gradeScale = [];
        foreach ($grades as $grade) {
            $gradeScale[$grade->grade] = [
                'min' => $grade->min_score,
                'comment' => $grade->comment
            ];
        }

        return $gradeScale;
    }

    /**
     * Get all phase configurations
     *
     * @return array
     */
    public function getAllPhaseConfigs(): array
    {
        $phases = PhaseConfig::all();
        $result = [];

        foreach ($phases as $phase) {
            $result[$phase->phase_name] = [
                'msk_theory' => $phase->msk_theory_config,
                'msk_practical' => $phase->msk_practical_config,
                'character_22cr' => $phase->character_22cr_config,
                'character_pr' => $phase->character_pr_config,
                'passing_criteria' => $phase->passing_criteria,
                'max_scores' => $phase->max_scores,
                'is_active' => $phase->is_active
            ];
        }

        return $result;
    }
}