<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\LogsActivity;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Validation\ValidationException;

class Assessment extends Model
{
    use HasFactory;
    use LogsActivity; 

    protected $fillable = [
        'admission_id', 
        'subject_id', 
        'marks', 
        'type', 
        'month',
        'weight', 
        'bn', 
        'coy', 
        'pl', 
        'sec', 
        'phase',
        'assessment_type',
        'assessment_group',
        'week',
        'exercises',
        'f1', 'f2', 'f3', 'f4', 'f5',
        'fx1', 'fx2', 'fx3', 'fx4', 'fx5',
        'sum_f', 'sum_fx',
        'mean_rating',
        'phase_weight_interval',
        'individual_cres',
        'code_name'
    ];

    /**
     * The fields that make a record unique
     */
    const UNIQUE_FIELDS = [
        'admission_id',
        'phase',
        'assessment_type',
        'assessment_group',
        'type',
        'subject_id',
        'code_name',
        'exercises',
        'bn',
        'coy',
        'pl',
        'sec',
        'week',
        'month'
    ];

    /**
     * Boot the model - prevent duplicates at the model level
     */
    protected static function boot()
    {
        parent::boot();

        // Prevent creating duplicates
        static::creating(function ($assessment) {
            $assessment->preventDuplicate();
        });

        // Prevent updating to create duplicates
        static::updating(function ($assessment) {
            $assessment->preventDuplicate(true);
        });

        // Add global scope to ensure we never get duplicates in queries
        static::addGlobalScope('no_duplicates', function (Builder $builder) {
            // This doesn't prevent duplicates, but ensures we handle them
        });
    }

    /**
     * Prevent duplicate records
     */
    protected function preventDuplicate($isUpdate = false)
    {
        $query = self::where('admission_id', $this->admission_id)
            ->where('phase', $this->phase)
            ->where('assessment_type', $this->assessment_type)
            ->where('assessment_group', $this->assessment_group)
            ->where('type', $this->type);

        // Handle nullable fields
        foreach (['subject_id', 'code_name', 'exercises', 'bn', 'coy', 'pl', 'sec', 'week', 'month'] as $field) {
            if ($this->{$field} !== null && $this->{$field} !== '') {
                $query->where($field, $this->{$field});
            } else {
                $query->whereNull($field);
            }
        }

        // Exclude current record if updating
        if ($isUpdate && $this->id) {
            $query->where('id', '!=', $this->id);
        }

        if ($query->exists()) {
            $fields = $this->formatDuplicateFields();
            throw ValidationException::withMessages([
                'duplicate' => 'A duplicate assessment already exists. ' . implode(', ', $fields)
            ]);
        }
    }

    /**
     * Format duplicate fields for error message
     */
    protected function formatDuplicateFields(): array
    {
        $fields = [];
        $fieldLabels = [
            'admission_id' => 'Student',
            'phase' => 'Phase',
            'assessment_type' => 'Assessment Type',
            'assessment_group' => 'Assessment Group',
            'type' => 'Type',
            'subject_id' => 'Subject',
            'code_name' => 'Code Name',
            'exercises' => 'Exercises',
            'bn' => 'BN',
            'coy' => 'COY',
            'pl' => 'PL',
            'sec' => 'SEC',
            'week' => 'Week',
            'month' => 'Month'
        ];

        foreach (self::UNIQUE_FIELDS as $field) {
            if ($this->{$field} !== null && $this->{$field} !== '') {
                $label = $fieldLabels[$field] ?? $field;
                $fields[] = "$label: {$this->{$field}}";
            }
        }

        return $fields;
    }

    /**
     * Find or create an assessment (prevents duplicates)
     */
    public static function findOrCreate(array $data): self
    {
        // Build unique key hash for caching/lookup
        $uniqueData = array_intersect_key($data, array_flip(self::UNIQUE_FIELDS));
        $hash = md5(json_encode($uniqueData));
        
        // Try to find existing
        $existing = self::where($uniqueData)->first();
        
        if ($existing) {
            return $existing;
        }
        
        return self::create($data);
    }

    /**
     * Update or create (prevents duplicates)
     */
    public static function updateOrCreateUnique(array $attributes, array $values = []): self
    {
        // Extract unique fields
        $uniqueData = array_intersect_key($attributes, array_flip(self::UNIQUE_FIELDS));
        
        // Find existing
        $existing = self::where($uniqueData)->first();
        
        if ($existing) {
            $existing->update($values);
            return $existing;
        }
        
        return self::create(array_merge($attributes, $values));
    }

    // ... rest of your existing methods (relationships, scopes, etc.) ...

    public function student()
    {
        return $this->belongsTo(Student::class, 'admission_id');
    }

    public function entries()
    {
        return $this->hasMany(AssessmentEntry::class, 'assessment_id');
    }

    public function admission()
    {
        return $this->belongsTo(Admission::class);
    }

    public function subject()
    {
        return $this->belongsTo(Subject::class);
    }

    public function getWeightedScoreAttribute()
    {
        return ($this->marks / 100) * $this->weight;
    }

    public function phase() {
        return $this->belongsTo(Phase::class);
    }



    public function scopeWithValidAdmissions($query)
    {
        return $query->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
                     ->whereNull('admissions.deleted_at');
    }
    
    public function scopeForActivePhase($query)
    {
        $phase = Phase::where('is_active', 1)->first();
        if ($phase) {
            return $query->where('assessments.phase', $phase->name);
        }
        return $query;
    }
}