<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Student;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Activitylog\LogOptions;
use App\Traits\LogsActivity;
use Illuminate\Database\Eloquent\Builder;
class Admission extends Model
{

    use SoftDeletes;
    use HasFactory;

    use LogsActivity;

    protected $fillable = [
        'name', 'rank', 'school_id', 'course_id', 'intake_id',
        'email', 'phone_number', 'sex', 'bn', 'coy', 'pl', 'sec',
        'image', 'nin', 'is_non_ugandan', 'army_number',
        
        'is_decommissioned',
        'decommission_date',
        'decommission_reason',
        'decommission_notes',
        'decommissioned_by',
        'phase',
        'home_district', 'village', 'father_name', 'father_phone',
        'next_kin_phone', 'next_kin_rel',
        'mother_name', 'mother_phone', 'spouse_name', 'spouse_phone',
        'education_level', 'next_kin', 'shoe_size', 'year_admission',
        'country_origin', 'military_id', 'created_by','deleted_at'
    ];

        // Customize log name (useful for filtering logs)
        protected static $logName = 'admission';

        // Log only the changed attributes
        protected static $logOnlyDirty = true;

        // Prevent logging same values
        protected static $submitEmptyLogs = false;

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
        ->logOnly(['name', 'admission_id']);
        // Chain fluent methods for configuration options
    }

    public function decommissionedBy()
    {
        return $this->belongsTo(User::class, 'decommissioned_by');
    }

    public function calculateTotal()
    {
        return $this->msk_theory +
               $this->msk_practical +
               $this->character_assessment_22cr +
               $this->character_assessment_pr;
    }

    public function calculateGrade()
    {
        $total = $this->calculateTotal();

        if ($total >= 80) return 'A';
        if ($total >= 70) return 'B';
        if ($total >= 60) return 'C';
        if ($total >= 50) return 'D';
        return 'F';
    }

    public function isPassed()
    {
        return $this->calculateTotal() >= 50;
    }
    // Relationships
    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function students()
    {
        return $this->hasMany(Student::class);
    }

    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function course()
    {
        return $this->belongsTo(Course::class);
    }

    public function intake()
    {
        return $this->belongsTo(Intake::class);
    }

    public function subjects(): BelongsToMany
    {
        return $this->belongsToMany(Subject::class, 'admission_subject', 'admission_id', 'subject_id')
                    ->withTimestamps();
    }

public function subject()
{
    return $this->belongsTo(Subject::class);
}

public function admission()
{
    return $this->belongsTo(Admission::class);
}


    public function hasSubject($subjectId): bool
    {
        return $this->subjects()->where('subjects.id', $subjectId)->exists();
    }

    public function armyRecord()
    {
        return $this->hasOne(ArmyRecord::class, 'ArmyNumber', 'army_number');
    }


    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by', 'id');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by', 'id');
    }

    public function receivedRatings()
{
    return $this->hasMany(PR::class, 'army_number', 'army_number');
}
    // Boot Method to Handle Admission ID & Student Existence
    protected static function boot()
    {
        parent::boot();

// static::addGlobalScope('active_intake', function (Builder $builder) {
//     $intake = \App\Models\Intake::where('is_active', 1)->first();
//     if ($intake) {
//         $builder->where('intake_id', $intake->id);
//     }
// });

static::addGlobalScope('non_decommissioned', function (Builder $builder) {
            $builder->where('is_decommissioned', '!=', 1)
                   ->orWhereNull('is_decommissioned');
        });

        static::creating(function ($admission) {
            // Ensure a student exists before proceeding
            $student = Student::firstOrCreate(
                ['army_number' => $admission->army_number],
                ['name' => $admission->name, 'rank' => $admission->rank]
            );

            $admission->student_id = $student->id;

            // Check if the student has already been admitted
            $exists = Admission::where('army_number', $admission->army_number)
                                ->where('course_id', $admission->course_id)
                                ->exists();

            if ($exists) {
                throw new \Exception('This student has already been admitted to this course.');
            }

            // Generate admission ID only if it's not already set
            if (empty($admission->admission_id)) {
                // Set year_admission if not set
                if (empty($admission->year_admission)) {
                    $admission->year_admission = now()->year;
                }

                $year = substr($admission->year_admission, -2); // Get last 2 digits of the year

                // Get the latest count for the specific year
                $latestAdmission = Admission::where('year_admission', $admission->year_admission)
                                          ->orderBy('id', 'desc')
                                          ->first();

                $count = $latestAdmission ? (int)substr($latestAdmission->admission_id, -6) + 1 : 1;
                $formattedCount = str_pad($count, 6, '0', STR_PAD_LEFT);

                // Fetch school short name
                $school = \App\Models\School::find($admission->school_id);
                $schoolCode = $school ? strtoupper(substr($school->name, 0, 4)) : 'BTMS';

                // Generate admission_id
                $admission->admission_id = "{$schoolCode}-{$year}-MH-{$formattedCount}";
            }
        });
    }

    public function crAssessments()
    {
        return $this->hasMany(CRAssessment2::class);
    }

    // Add this method to your Admission model
public function scopeActive($query)
{
    return $query->where('is_decommissioned', '!=', 1)
                ->orWhereNull('is_decommissioned');
}

// Then use it in your queries:
// Admission::active()->get();

public function scopeSortable($query, $request)
{
    $perPage = $request->input('per_page', 25);
    
    $sort = $request->input('sort', 'admission_id');
    $direction = $request->input('direction', 'asc');
    
    $allowedSorts = ['admission_id', 'name', 'army_number', 'rank'];
    
    // Ensure the column is allowed
    if (!in_array($sort, $allowedSorts)) {
        $sort = 'admission_id';
    }
    
    // Ensure direction is valid
    if (!in_array($direction, ['asc', 'desc'])) {
        $direction = 'asc';
    }
    
    // Exclude decommissioned records
    $query->where('is_decommissioned', '!=', 1)
          ->orWhereNull('is_decommissioned');
    
    // Special case for sorting `admission_id`
    if ($sort === 'admission_id') {
        return $query->orderByRaw("CAST(SUBSTRING_INDEX(admission_id, '-', -1) AS UNSIGNED) $direction");
    }
    
    return $query->orderBy($sort, $direction);
}
    
}
