<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class EnsureSsoAuthenticated
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Check if user is authenticated
        if (!Auth::check()) {
            Log::warning('🚫 Unauthenticated access attempt', [
                'url' => $request->fullUrl(),
                'ip' => $request->ip()
            ]);
            
            // Redirect to SSO login (home route)
            return redirect()->route('home')
                ->with('error', 'Please login through MCOP to access this system.');
        }
        
        // Check if SSO authenticated
        if (!session('sso_authenticated')) {
            Log::warning('🚫 Non-SSO authenticated user', [
                'user_id' => Auth::id(),
                'url' => $request->fullUrl()
            ]);
            
            // Logout and redirect to MCOP
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            
            return redirect()->route('home')
                ->with('error', 'SSO authentication required. Please login through MCOP.');
        }
        
        // Check if user is active (if using is_active field)
        $user = Auth::user();
        if (property_exists($user, 'is_active') && !$user->is_active) {
            Log::warning('🚫 Inactive user access attempt', [
                'user_id' => $user->id,
                'email' => $user->email
            ]);
            
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            
            return redirect()->route('home')
                ->with('error', 'Your account is inactive. Contact administrator.');
        }
        
        // Check if SSO is enabled for this user
        if (property_exists($user, 'sso_enabled') && !$user->sso_enabled) {
            Log::warning('🚫 SSO disabled user access attempt', [
                'user_id' => $user->id,
                'email' => $user->email
            ]);
            
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            
            return redirect()->route('home')
                ->with('error', 'SSO access is not enabled for your account.');
        }
        
        // Optional: Check token expiration (if you want to enforce timeout)
        $ssoLoginTime = session('sso_login_time');
        if ($ssoLoginTime) {
            $hoursSinceLogin = now()->diffInHours($ssoLoginTime);
            $maxSessionHours = config('session.lifetime', 120) / 60; // Convert minutes to hours
            
            if ($hoursSinceLogin > $maxSessionHours) {
                Log::info('⏰ SSO session expired', [
                    'user_id' => Auth::id(),
                    'hours_since_login' => $hoursSinceLogin
                ]);
                
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                
                return redirect()->route('home')
                    ->with('error', 'Your session has expired. Please login again.');
            }
        }
        
        return $next($request);
    }
}