<?php

namespace App\Http\Controllers;

use App\Models\Remark;
use App\Models\Admission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class RemarkController extends Controller
{
    /**
     * Show the form for creating a new remark.
     * 
     * @param int $admissionId
     * @return \Illuminate\View\View
     */
    public function create($admissionId)
    {
        $admission = Admission::findOrFail($admissionId);
        
        // Fetch all remarks for the specific admission to show below
        $remarks = Remark::where('admission_id', $admissionId)->get();


        
        return view('remarks.create', compact('admission','remarks'));
    }
    
    /**
     * Fetch remarks for a specific admission (AJAX endpoint)
     * 
     * @param int $admissionId
     * @return \Illuminate\Http\JsonResponse
     */
    public function fetchRemarks($admissionId)
    {
        $remarks = Remark::where('admission_id', $admissionId)->get();
        return response()->json($remarks);
    }
    
    /**
     * Store a newly created remark in storage.
     * 
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
  public function store(Request $request)
{
    // Initialize uploaded files array
    $uploadedFiles = [];
    $newUploadedFiles = []; // Added this line to fix undefined variable
    
    // Start a database transaction
    DB::beginTransaction();
    
    try {
        // Modified validation to handle phase or phase_name
        $validated = $request->validate([
            'admission_id' => 'required|exists:admissions,id',
            'phase' => 'required_without:phase_name|nullable|in:1,2,3',
            'phase_name' => 'required_without:phase|nullable|string', // Added phase_name validation
            'pl_comd_remarks' => 'nullable|string|max:5000',
            'oc_remarks' => 'nullable|string|max:5000',
            'ci_remarks' => 'nullable|string|max:5000',
            'pl_comd_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
            'oc_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
         
        ]);
        
        $data = $validated;
        
        // If we have a phase_name but not a phase number, handle accordingly
        if (empty($data['phase']) && !empty($data['phase_name'])) {
            // Map the phase_name to a number if needed
            // Or keep the phase_name and don't set phase
            // Uncomment the needed approach:
            
            // Option 1: Map phase_name to phase number
            /*
            $phaseMap = [
                'phase1_name' => 1,
                'phase2_name' => 2,
                'phase3_name' => 3,
            ];
            $data['phase'] = $phaseMap[$data['phase_name']] ?? null;
            */
            
            // Option 2: Just use the phase_name as is
            // No action needed
        }
        
        // Handle signature file uploads
        $signatureFields = [
            'pl_comd_signature', 
            'oc_signature', 
            'ci_signature', 
            'comdt_signature', 
            'student_signature'
        ];

        foreach ($signatureFields as $field) {
            if ($request->hasFile($field) && $request->file($field)->isValid()) {
                try {
                    $file = $request->file($field);

                    $filename = $data['admission_id'] . '_' . $field . '_' . time() . '_' . \Illuminate\Support\Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->extension();

                    // Define custom public directory
                    $customPath = public_path('signatures');
                    if (!file_exists($customPath)) {
                        mkdir($customPath, 0777, true);
                    }

                    // Move the file to /public/signatures
                    $file->move($customPath, $filename);

                    // Save relative path (relative to public/) for DB
                    $data[$field] = 'signatures/' . $filename;

                    // Track new file for cleanup if needed
                    $newUploadedFiles[] = $data[$field];

                    // Optional: Optimize image if Intervention/Image is available
                    if (class_exists('Intervention\Image\Facades\Image')) {
                        $img = \Intervention\Image\Facades\Image::make(public_path($data[$field]));
                        $img->resize(600, null, function ($constraint) {
                            $constraint->aspectRatio();
                            $constraint->upsize();
                        });
                        $img->save();
                    }

                    Log::info('Signature stored successfully', [
                        'field' => $field,
                        'path' => $data[$field],
                        'admission_id' => $data['admission_id']
                    ]);
                } catch (\Exception $e) {
                    Log::error('File upload failed', [
                        'field' => $field,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw new \Exception('Error uploading signature: ' . $field . '. ' . $e->getMessage());
                }
            }
        }

        // Create the remark
        $remark = Remark::create($data);
        
        // Commit the transaction
        DB::commit();
        
        // Log success
        Log::info('Remark created successfully', [
            'remark_id' => $remark->id,
            'admission_id' => $validated['admission_id'],
            'created_by' => auth()->id() ?? 'unknown'
        ]);
        
        // For AJAX requests, return JSON response
        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Remark created successfully with signatures.',
                'remark' => $remark
            ]);
        }
        
        // Get the admission for the redirect
        $admission = Admission::findOrFail($validated['admission_id']);
        
        // Fetch all remarks for the specific admission
        $remarks = Remark::where('admission_id', $validated['admission_id'])->get();
        
        // Redirect back to the create page with success message and updated data
        return redirect()->route('remarks.create', ['admissionId' => $validated['admission_id']])
            ->with('success', 'Remark created successfully with signatures.');
            
    } catch (\Illuminate\Validation\ValidationException $e) {
        // Rollback the transaction
        DB::rollBack();
        
        // Clean up any uploaded files if they exist
        if (!empty($newUploadedFiles)) {
            $this->cleanupUploadedFiles($newUploadedFiles);
        }
        
        // Re-throw validation exception for proper handling
        throw $e;
        
    } catch (\Exception $e) {
        // Rollback the transaction
        DB::rollBack();
        
        // Clean up any uploaded files if they exist
        if (!empty($newUploadedFiles)) {
            $this->cleanupUploadedFiles($newUploadedFiles);
        }
        
        Log::error('Remark creation failed', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'request_data' => $request->except($signatureFields)
        ]);
        
        // For AJAX requests, return JSON response
        if ($request->expectsJson()) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create remark',
                'error' => $e->getMessage()
            ], 500);
        }
        
        return back()
            ->withInput()
            ->withErrors(['error' => 'Failed to create remark. ' . $e->getMessage()]);
    }
}
    public function edit($id)
    {
        $remark = Remark::findOrFail($id);
        $admission = Admission::findOrFail($remark->admission_id);

        return view('remarks.edit', compact('remark', 'admission'));
    }

    /**
 * Update the specified remark in storage.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $id
 * @return \Illuminate\Http\Response
 */
public function update(Request $request, $id)
{
    $remark = Remark::findOrFail($id);

    // Validate input
    $validatedData = $request->validate([
           
                'pl_comd_remarks' => 'nullable|string|max:5000',
                'oc_remarks' => 'nullable|string|max:5000',
                'ci_remarks' => 'nullable|string|max:5000',
                'pl_comd_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
                'oc_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
                'ci_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
    ]);

    DB::beginTransaction();

    try {
        $data = $validatedData;
        $signatureFields = [
            'pl_comd_signature', 
            'oc_signature', 
          
      
       
 

        ];

        $newUploadedFiles = [];

        foreach ($signatureFields as $field) {
            if ($request->hasFile($field) && $request->file($field)->isValid()) {
                try {
                    // Delete old file if exists
                    if ($remark->$field && file_exists(public_path($remark->$field))) {
                        unlink(public_path($remark->$field));
                    }

                    $file = $request->file($field);
                    $filename = $remark->admission_id . '_' . $field . '_' . time() . '_' . \Illuminate\Support\Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->extension();

                    // Define custom public directory
                    $customPath = public_path('signatures');
                    if (!file_exists($customPath)) {
                        mkdir($customPath, 0777, true);
                    }

                    // Move the file to /public/signatures
                    $file->move($customPath, $filename);

                    // Save relative path (relative to public/)
                    $data[$field] = 'signatures/' . $filename;

                    // Track new uploaded file for cleanup if needed
                    $newUploadedFiles[] = $data[$field];

                    // Optional: Optimize image if Intervention/Image is available
                    if (class_exists('Intervention\Image\Facades\Image')) {
                        $img = \Intervention\Image\Facades\Image::make(public_path($data[$field]));
                        $img->resize(600, null, function ($constraint) {
                            $constraint->aspectRatio();
                            $constraint->upsize();
                        });
                        $img->save();
                    }

                    Log::info('Signature updated successfully', [
                        'field' => $field,
                        'path' => $data[$field],
                        'remark_id' => $id
                    ]);
                } catch (\Exception $e) {
                    Log::error('File update failed', [
                        'field' => $field,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw new \Exception('Error updating signature: ' . $field . '. ' . $e->getMessage());
                }
            } else {
                unset($data[$field]);
            }
        }

        // Update database
        $remark->update($data);

        DB::commit();

        return  redirect()->back()->with('success', 'Remark updated successfully');

        // return redirect()->route('remarks.create')->with('success', 'Remark updated successfully');
    } catch (\Exception $e) {
        DB::rollBack();

        // Cleanup any newly uploaded files
        $this->cleanupUploadedFiles($newUploadedFiles);

        Log::error('Remark update failed', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        return redirect()->back()->withErrors(['error' => 'Failed to update remark: ' . $e->getMessage()])->withInput();
    }
}

// Helper function for cleanup
protected function cleanupUploadedFiles($files)
{
    foreach ($files as $file) {
        $filePath = public_path($file);
        if (file_exists($filePath)) {
            unlink($filePath);
        }
    }
}

    /**
     * Remove a signature file.
     * 
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\Remark $remark
     * @param string $field
     * @return \Illuminate\Http\Response
     */
    public function removeSignature(Request $request, Remark $remark, $field)
    {
        // Check if field is valid
        $validFields = [
            'pl_comd_signature', 
            'oc_signature', 
            'ci_signature', 
            'comdt_signature', 
            'student_signature'
        ];
        
        if (!in_array($field, $validFields)) {
            return back()->with('error', 'Invalid signature field.');
        }
        
        // Delete the file if it exists
        if ($remark->$field && Storage::disk('public')->exists($remark->$field)) {
            Storage::disk('public')->delete($remark->$field);
        }
        
        // Update the database record
        $remark->$field = null;
        $remark->save();
        
        return back()->with('success', 'Signature removed successfully.');
    }
    
    /**
     * Display the specified remark.
     * 
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $remark = Remark::findOrFail($id);
        return view('remarks.show', compact('remark'));
    }

    public function destroy($id)
    {
        $remark = Remark::findOrFail($id);
        
        // Delete associated files
        $signatureFields = [
            'pl_comd_signature', 
            'oc_signature', 
            'ci_signature', 
            'comdt_signature', 
            'student_signature'
        ];
        
        foreach ($signatureFields as $field) {
            if ($remark->$field && Storage::disk('public')->exists($remark->$field)) {
                Storage::disk('public')->delete($remark->$field);
            }
        }
        
        // Delete the remark
        $remark->delete();
        
        return redirect()->route('remarks.create', ['admissionId' => $remark->admission_id])
            ->with('success', 'Remark deleted successfully.');
    }
    
}