<?php

namespace App\Http\Controllers;

use App\Models\Registration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class RegistrationController extends Controller
{
    public function index()
    {
        $registrations = Registration::latest()->paginate(10);
        return view('registrations.index', compact('registrations'));
    }

    public function create()
    {
        return view('registrations.create');
    }

    public function store(Request $request)
    {


        // 'kin_number','relationship_with_kin'
        $validated = $request->validate([
            'kin_number' => 'nullable',
             'shoe_size'=> 'nullable',
             'email'=> 'nullable',
            'relationship_with_kin' => 'nullable',
            'army_number' => 'required|unique:registrations',
            'name' => 'required',
            'photo' => 'image|mimes:jpeg,png,jpg',
            'nin' => 'required|unique:registrations',
            'dob' => 'required|date',
            'sex' => 'required|in:Male,Female',
            'phone' => 'required',
            'unit' => 'nullable',
            'father_name' => 'nullable',
            'father_phone' => 'nullable',
            'mother_name' => 'nullable',
            'mother_phone' => 'nullable',
            'spouse_name' => 'nullable',
            'spouse_phone' => 'nullable',
            'education_level' => 'nullable',
            'next_kin' => 'nullable',
        ]);
    
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
            $validated['photo'] = $photoPath;
        }
    
        // Get the current year
        $currentYear = date('Y');
    
        // Get the last registration number for this year
        $lastRegistration = Registration::whereYear('created_at', $currentYear)
                                          ->latest()
                                          ->first();
    
        // Generate the next sequential number
        $nextNumber = $lastRegistration ? (int) substr($lastRegistration->registration_number, -5) + 1 : 1;
    
        // Format the registration number as REG-YYYY-00001
        $validated['registration_number'] = 'REG-' . $currentYear . '-' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    
        Registration::create($validated);
    
        return redirect()->route('registrations.index')
            ->with('success', 'Registration created successfully.');
    }
    

    public function edit(Registration $registration)
    {
        return view('registrations.edit', compact('registration'));
    }

    public function update(Request $request, Registration $registration)
    {
        $validated = $request->validate([
            'kin_number' => 'nullable',
            'shoe_size'=> 'nullable',
            'email'=> 'nullable',
           'relationship_with_kin' => 'nullable',
    
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
    
            'phone' => 'required',
            'unit' => 'nullable',
            'father_name' => 'nullable',
            'father_phone' => 'nullable',
            'mother_name' => 'nullable',
            'mother_phone' => 'nullable',
            'spouse_name' => 'nullable',
            'spouse_phone' => 'nullable',
            'education_level' => 'nullable',
            'next_kin' => 'nullable',
        ]);

        if ($request->hasFile('photo')) {
            // Delete old photo
            if ($registration->photo) {
                Storage::disk('public')->delete($registration->photo);
            }
            $photoPath = $request->file('photo')->store('photos', 'public');
            $validated['photo'] = $photoPath;
        }

        $registration->update($validated);

        return redirect()->route('registrations.index')
            ->with('success', 'Registration updated successfully.');
    }

    public function destroy(Registration $registration)
    {
        if ($registration->photo) {
            Storage::disk('public')->delete($registration->photo);
        }
        
        $registration->delete();

        return redirect()->route('registrations.index')
            ->with('success', 'Registration deleted successfully.');
    }


    public function show(Registration $registration)
    {
        return view('registrations.show', compact('registration'));
    }
}