<?php

namespace App\Http\Controllers;

use App\Models\Bn;
use App\Models\Coy;
use App\Models\Pl;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PlController extends Controller
{
    public function index()
    {
        try {
            $pl = Pl::all();
        } catch (\Exception $e) {
            Log::error('Error fetching platoons: ' . $e->getMessage());
            return redirect()->route('bn.index')->with('error', 'Failed to fetch platoons.');
        }
        return view('bn.index', compact('pl'));
    }

    public function create()
    {
        try {
            $bns = Bn::all();
            $coys = Coy::all();
            $pls = Pl::all();
        } catch (\Exception $e) {
            Log::error('Error fetching companies: ' . $e->getMessage());
            return redirect()->route('bn.index')->with('error', 'Failed to fetch companies.');
        }
        return view('pl.create', compact('coys','bns','pls'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'coy_id' => 'required|exists:coys,id',
            'bn_id' => 'required|exists:bns,id',
            'pl_sgt' => 'nullable|string|max:255',
            'comd_name' => 'nullable|string|max:255',
            'comd_rank' => 'nullable|string|max:255',
            'pl_sgt_rank' => 'required|string|max:255',
            'comd_initials' => 'nullable|string|max:255',
        ]);

        DB::beginTransaction();
        try {
            Pl::create($validated);
            DB::commit();
            return redirect()->route('bn.index')->with('success', 'Platoon created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating platoon: ' . $e->getMessage());
            return redirect()->route('bn.index')->with('error', 'Failed to create platoon.');
        }
    }

    public function edit(Pl $pl)
    {
        try {
            $coys = Coy::all();
            $bns = Bn::all();
        } catch (\Exception $e) {
            Log::error('Error fetching companies: ' . $e->getMessage());
            return redirect()->route('bn.index')->with('error', 'Failed to fetch companies.');
        }
        return view('pl.edit', compact('pl', 'coys','bns'));
    }

    public function update(Request $request, Pl $pl)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'coy_id' => 'required|exists:coys,id',
            'bn_id' => 'required|exists:bns,id',
            'comd_name' => 'nullable|string|max:255',
            'comd_rank' => 'nullable|string|max:255',
            'pl_sgt' => 'nullable|string|max:255',
            'comd_initials' => 'nullable|string|max:255',
            'pl_sgt_rank' => 'required|string|max:255',
        ]);

        DB::beginTransaction();
        try {
            $pl->update($validated);
            DB::commit();
            return redirect()->route('bn.index')->with('success', 'Platoon updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating platoon: ' . $e->getMessage());
            return redirect()->route('bn.index')->with('error', 'Failed to update platoon.');
        }
    }

    public function destroy(Pl $pl)
    {
        DB::beginTransaction();
        try {
            $pl->delete();
            DB::commit();
            return redirect()->route('bn.index')->with('success', 'Platoon deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting platoon: ' . $e->getMessage());
            return redirect()->route('bn.index')->with('error', 'Failed to delete platoon.');
        }
    }
    public function getCoysByBn(Request $request)
    {
        $bn_id = $request->get('bn_id');
        
        // Add where clause to only get Coys belonging to the selected Bn
        $coys = Coy::where('bn_id', $bn_id)->get();
        
        return response()->json($coys);
    }

    public function getPlsByCoy(Request $request)
{
    $coy_id = $request->get('coy_id');
    
    // Get platoons belonging to the selected company
    $pls = Pl::where('coy_id', $coy_id)->get();
    
    return response()->json($pls);
}
    
    public function getPlsByBnCoy(Request $request)
    {
        try {
            $bn_id = $request->bn_id;
            $coy_id = $request->coy_id;
    
            // Validate inputs
            if (!$bn_id || !$coy_id) {
                return response()->json(['error' => 'Battalion ID and Company ID are required'], 400);
            }
    
            $pls = Pl::where('bn_id', $bn_id)
                   ->where('coy_id', $coy_id)
                   ->get(['id', 'name']);  // Only select needed fields for performance
    
            return response()->json($pls);
        } catch (\Exception $e) {
            Log::error('Error fetching platoons: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch platoons.'], 500);
        }
    }

}
