<?php

namespace App\Http\Controllers;

use App\Models\PhaseConfig;
use App\Models\GradeScale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use App\Models\Phase;
use Illuminate\Support\Facades\Cache;
use App\Models\Admission;
use App\Models\GpvReport;
use App\Models\GpvReportDetail;
use App\Models\Intake;
use App\Models\School;
use App\Models\Course;
use App\Models\Bn;
use App\Models\Coy;
use App\Models\Pl;
use App\Http\Controllers\StudentPerformanceController;

class PhaseConfigController extends Controller
{
    /**
     * Display a listing of all phase configurations
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $phases = PhaseConfig::all();
        $gradeScales = GradeScale::orderBy('min_score', 'desc')->get();
        
        return view('phase-config.index', compact('phases', 'gradeScales'));
    }
    
    /**
     * Show the form for creating a new phase configuration
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('phase-config.create');
    }
    
    /**
     * Store a newly created phase configuration in storage
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phase_name' => 'required|string|max:255|unique:phase_configs',
            // Theory
            'theory_max_score' => 'required|numeric|min:0',
            'theory_pass_mark' => 'required|numeric|min:0|lte:theory_max_score',
            // Practical
            'practical_max_score' => 'required|numeric|min:0',
            'practical_pass_mark' => 'required|numeric|min:0|lte:practical_max_score',
            // 22CR
            'cr22_max_score' => 'required|numeric|min:0',
            'cr22_pass_mark' => 'required|numeric|min:0|lte:cr22_max_score',
            // PR
            'pr_max_score' => 'required|numeric|min:0',
            'pr_pass_mark' => 'required|numeric|min:0|lte:pr_max_score',
            'is_active' => 'boolean'
        ]);
        
        if ($validator->fails()) {
            return redirect()->route('phase-config.create')
                ->withErrors($validator)
                ->withInput();
        }
        
        // If setting this phase as active, deactivate all others
        if ($request->has('is_active') && $request->is_active) {
            PhaseConfig::where('is_active', true)->update(['is_active' => false]);
        }
        
        // Create the phase configuration
        $phaseConfig = new PhaseConfig();
        $phaseConfig->phase_name = $request->phase_name;
        
        // Set the configurations
        $phaseConfig->msk_theory_config = [
            'max_score' => (float)$request->theory_max_score,
            'pass_mark' => (float)$request->theory_pass_mark,
            'obtained' => 0
        ];
        
        $phaseConfig->msk_practical_config = [
            'max_score' => (float)$request->practical_max_score,
            'pass_mark' => (float)$request->practical_pass_mark,
            'obtained' => 0
        ];
        
        $phaseConfig->character_22cr_config = [
            'max_score' => (float)$request->cr22_max_score,
            'pass_mark' => (float)$request->cr22_pass_mark,
            'obtained' => 0
        ];
        
        $phaseConfig->character_pr_config = [
            'max_score' => (float)$request->pr_max_score,
            'pass_mark' => (float)$request->pr_pass_mark,
            'obtained' => 0
        ];
        
        // Set passing criteria
        $phaseConfig->passing_criteria = [
            'theory' => (float)$request->theory_pass_mark,
            'practical' => (float)$request->practical_pass_mark,
            '22cr' => (float)$request->cr22_pass_mark,
            'pr' => (float)$request->pr_pass_mark
        ];
        
        // Set max scores
        $phaseConfig->max_scores = [
            'theory' => (float)$request->theory_max_score,
            'practical' => (float)$request->practical_max_score,
            '22cr' => (float)$request->cr22_max_score,
            'pr' => (float)$request->pr_max_score
        ];
        
        $phaseConfig->is_active = $request->has('is_active') ? (bool)$request->is_active : false;
        $phaseConfig->save();
        
        return redirect()->route('phase-config.index')
            ->with('success', 'Phase configuration created successfully.');
    }
    
    /**
     * Show the form for editing the specified phase configuration
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $phase = PhaseConfig::findOrFail($id);
    
        return view('phase-config.edit', compact('phase'));
    }
    
    /**
     * Update the specified phase configuration in storage
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $phaseConfig = PhaseConfig::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'phase_name' => 'required|string|max:255|unique:phase_configs,phase_name,' . $id,
            'theory_max_score' => 'required|numeric|min:0',
            'theory_pass_mark' => 'required|numeric|min:0|lte:theory_max_score',
            'practical_max_score' => 'required|numeric|min:0',
            'practical_pass_mark' => 'required|numeric|min:0|lte:practical_max_score',
            'cr22_max_score' => 'required|numeric|min:0',
            'cr22_pass_mark' => 'required|numeric|min:0|lte:cr22_max_score',
            'pr_max_score' => 'required|numeric|min:0',
            'pr_pass_mark' => 'required|numeric|min:0|lte:pr_max_score',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->route('phase-config.edit', $id)
                ->withErrors($validator)
                ->withInput();
        }

        $activatingThis = $request->has('is_active') && $request->is_active;
        $previousActivePhase = null;
        $targetPhase = Phase::find($id);

        if ($activatingThis && $targetPhase) {
            $previousActivePhase = Phase::where('is_active', 1)
                                        ->where('id', '!=', $id)
                                        ->first();
        }

        DB::beginTransaction();

        try {
            // --- ARCHIVE THE PREVIOUS ACTIVE PHASE (if switching) ---
            if ($previousActivePhase && $targetPhase) {
                $this->archivePhaseData($previousActivePhase, $targetPhase);
            }

            $isChangingActivePhase = $activatingThis;

            // Deactivate all other phases if we are activating this one
            if ($isChangingActivePhase) {
                PhaseConfig::where('id', '!=', $id)->update(['is_active' => false]);
                Phase::where('id', '!=', $id)->update(['is_active' => false]);
            }

            // Update PhaseConfig
            $phaseConfig->phase_name = $request->phase_name;

            $phaseConfig->msk_theory_config = [
                'max_score' => (float)$request->theory_max_score,
                'pass_mark' => (float)$request->theory_pass_mark,
                'obtained' => 0
            ];

            $phaseConfig->msk_practical_config = [
                'max_score' => (float)$request->practical_max_score,
                'pass_mark' => (float)$request->practical_pass_mark,
                'obtained' => 0
            ];

            $phaseConfig->character_22cr_config = [
                'max_score' => (float)$request->cr22_max_score,
                'pass_mark' => (float)$request->cr22_pass_mark,
                'obtained' => 0
            ];

            $phaseConfig->character_pr_config = [
                'max_score' => (float)$request->pr_max_score,
                'pass_mark' => (float)$request->pr_pass_mark,
                'obtained' => 0
            ];

            $phaseConfig->passing_criteria = [
                'theory' => (float)$request->theory_pass_mark,
                'practical' => (float)$request->practical_pass_mark,
                '22cr' => (float)$request->cr22_pass_mark,
                'pr' => (float)$request->pr_pass_mark
            ];

            $phaseConfig->max_scores = [
                'theory' => (float)$request->theory_max_score,
                'practical' => (float)$request->practical_max_score,
                '22cr' => (float)$request->cr22_max_score,
                'pr' => (float)$request->pr_max_score
            ];

            $phaseConfig->is_active = $activatingThis;
            $phaseConfig->save();

            // Update or create corresponding Phase record
            $phase = Phase::find($id);

            if ($phase) {
                $phase->name = $request->phase_name;
                $phase->is_active = $activatingThis;
                $phase->cres_weight = (float)$request->cr22_max_score;
                $phase->pr_weight = (float)$request->pr_max_score;
                $phase->save();
            } else {
                $phase = Phase::create([
                    'id' => $id,
                    'name' => $request->phase_name,
                    'weeks' => 10,
                    'cres_weight' => (float)$request->cr22_max_score,
                    'pr_weight' => (float)$request->pr_max_score,
                    'is_active' => $activatingThis,
                ]);
            }

            DB::commit();

            // Clear cache if active phase changed
            if ($isChangingActivePhase) {
                Cache::forget('active_phase');
                Cache::remember('active_phase', 3600, function () use ($phase) {
                    return $phase ?? Phase::where('is_active', 1)->first();
                });
            }

            return redirect()->route('phase-config.index')
                ->with('success', 'Phase configuration and phase updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            // Check if it's the foreign key constraint error
            if (str_contains($e->getMessage(), 'CONSTRAINT `gpv_reports_intake_id_foreign`')) {
                return redirect()->route('phase-config.edit', $id)
                    ->withErrors(['error' => 'Cannot archive phase data: The intake record does not exist.'])
                    ->withInput();
            }
            
            return redirect()->route('phase-config.edit', $id)
                ->withErrors(['error' => 'Failed to update phase configuration: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Archive all admissions of the given old phase's intake into a GpvReport.
     *
     * @param Phase $oldPhase  The phase that is ending
     * @param Phase $newPhase  The phase that will become active
     * @return void
     */
    private function archivePhaseData(Phase $oldPhase, Phase $newPhase)
    {
        try {
            // Get the active intake
            $activeIntake = Intake::where('is_active', true)->first();
            
            if (!$activeIntake) {
                $activeIntake = Intake::latest()->first();
            }
            
            if (!$activeIntake) {
                \Log::warning('No intake found for phase archiving', [
                    'old_phase' => $oldPhase->name,
                    'new_phase' => $newPhase->name
                ]);
                return;
            }
            
            // Fetch all admissions belonging to the active intake
            $admissions = Admission::where('intake_id', $activeIntake->id)
                                   ->with(['student', 'course', 'school'])
                                   ->get();
            
            if ($admissions->isEmpty()) {
                \Log::info('No admissions to archive for intake', [
                    'intake_id' => $activeIntake->id
                ]);
                return;
            }

            // Create an instance of StudentPerformanceController to use its methods
            $studentPerformanceController = app(StudentPerformanceController::class);
            
            $studentsData = [];
            $uniqueSchoolId = null;
            $uniqueCourseId = null;
            $bnValues = [];
            $coyValues = [];
            $plValues = [];
            
            // Get max scores
            $maxScores = $this->getMaxScores();
            
            foreach ($admissions as $admission) {
                // Track the first school and course found
                if (!$uniqueSchoolId && $admission->school_id) {
                    $uniqueSchoolId = $admission->school_id;
                }
                if (!$uniqueCourseId && $admission->course_id) {
                    $uniqueCourseId = $admission->course_id;
                }
                
                // Track BN, COY, PL values
                if ($admission->bn && !in_array($admission->bn, $bnValues)) {
                    $bnValues[] = $admission->bn;
                }
                if ($admission->coy && !in_array($admission->coy, $coyValues)) {
                    $coyValues[] = $admission->coy;
                }
                if ($admission->pl && !in_array($admission->pl, $plValues)) {
                    $plValues[] = $admission->pl;
                }
                
                // Calculate scores
                $theoryScore = $admission->theory_score ?? 0;
                $practicalScore = $admission->practical_score ?? 0;
                $cr22Score = $admission->cr_22_score ?? 0;
                $prScore = $admission->pr_score ?? 0;
                
                // If scores are not stored, calculate them using the student performance controller
                if (!$admission->theory_score) {
                    $theoryScore = $studentPerformanceController->calculateTheoryScore(
                        $admission, 
                        $maxScores['theory'] ?? 40, 
                        $oldPhase->name
                    );
                }
                
                if (!$admission->practical_score) {
                    $practicalScore = $studentPerformanceController->calculatePracticalScore(
                        $admission->id, 
                        $oldPhase->name
                    );
                }
                
                if (!$admission->cr_22_score) {
                    $cr22Score = $studentPerformanceController->calculate22CRScore(
                        $admission, 
                        $maxScores['22cr'] ?? 15, 
                        $oldPhase->name
                    );
                }
                
                if (!$admission->pr_score) {
                    $prScore = $studentPerformanceController->calculatePrScore(
                        $admission, 
                        $maxScores['pr'] ?? 15, 
                        $oldPhase->name
                    );
                }
                
                $totalScore = $admission->total_score ?? ($theoryScore + $practicalScore + $cr22Score + $prScore);
                $grade = $admission->grade ?? $this->calculateGrade($totalScore);
                $comment = $admission->comment ?? $this->getComment($totalScore);
                $remarks = $admission->remarks ?? ($totalScore >= 50 ? 'PASSED' : 'FAILED');
                
                $studentsData[] = [
                    'student_id'       => $admission->id,
                    'admission_number' => $admission->admission_id ?? $admission->admission_number ?? '',
                    'army_number'      => $admission->army_number ?? optional($admission->student)->army_number ?? '',
                    'rank'             => $admission->rank ?? optional($admission->student)->rank ?? '',
                    'name'             => $admission->name ?? optional($admission->student)->name ?? '',
                    'school'           => optional($admission->school)->name ?? 'N/A',
                    'course'           => optional($admission->course)->name ?? 'N/A',
                    'theory'           => $theoryScore,
                    'practical'        => $practicalScore,
                    'cr_22'            => $cr22Score,
                    'pr'               => $prScore,
                    'total'            => $totalScore,
                    'grade'            => $grade,
                    'comment'          => $comment,
                    'remarks'          => $remarks,
                ];
            }

            // Get school and course names for better report naming
            $schoolName = $uniqueSchoolId ? School::find($uniqueSchoolId)->name : 'Multiple Schools';
            $courseName = $uniqueCourseId ? Course::find($uniqueCourseId)->name : 'Multiple Courses';
            
            // Create report name
            $reportName = 'GPV_' . $oldPhase->name .  
                          '_' . ($activeIntake->year ?? 'Intake_' . $activeIntake->id) . 
                          '_' . now()->format('Y-m-d_H-i-s');

            // Create a GpvReport record for this phase transition
            $archiveReport = GpvReport::create([
                'report_name'    => $reportName,
                'intake_id'      => $activeIntake->id,
                'school_id'      => $uniqueSchoolId,
                'course_id'      => $uniqueCourseId,
                'phase'          => $oldPhase->name,
                'bn'             => null,
                'coy'            => null,
                'pl'             => null,
                'grade'          => null,
                'army_number'    => null,
                'total_students' => count($studentsData),
                'filters'        => json_encode([
                    'archive_type'     => 'phase_transition',
                    'from_phase_id'    => $oldPhase->id,
                    'from_phase_name'  => $oldPhase->name,
                    'to_phase_id'      => $newPhase->id,
                    'to_phase_name'    => $newPhase->name,
                    'intake_id'        => $activeIntake->id,
                    'intake_name'      => $activeIntake->year ?? $activeIntake->name ?? 'Unknown',
                    'school_id'        => $uniqueSchoolId,
                    'school_name'      => $schoolName,
                    'course_id'        => $uniqueCourseId,
                    'course_name'      => $courseName,
                    'bn_values'        => $bnValues,
                    'coy_values'       => $coyValues,
                    'pl_values'        => $plValues,
                    'max_scores_used'  => $maxScores,
                    'archived_at'      => now()->toDateTimeString(),
                ]),
                'created_by'     => auth()->id() ?? 1,
            ]);

            // Insert each student as a detail record
            if (!empty($studentsData)) {
                $details = [];
                foreach ($studentsData as $student) {
                    $details[] = [
                        'gpv_report_id'    => $archiveReport->id,
                        'admission_id'     => $student['student_id'],
                        'admission_number' => $student['admission_number'],
                        'army_number'      => $student['army_number'],
                        'rank'             => $student['rank'],
                        'name'             => $student['name'],
                        'school'           => $student['school'],
                        'course'           => $student['course'],
                        'theory_score'     => $student['theory'],
                        'practical_score'  => $student['practical'],
                        'cr_22_score'      => $student['cr_22'],
                        'pr_score'         => $student['pr'],
                        'total_score'      => $student['total'],
                        'grade'            => $student['grade'],
                        'comment'          => $student['comment'],
                        'remarks'          => $student['remarks'],
                        'created_at'       => now(),
                        'updated_at'       => now(),
                    ];
                }
                
                // Insert in chunks
                foreach (array_chunk($details, 100) as $chunk) {
                    GpvReportDetail::insert($chunk);
                }
            }

            \Log::info('Phase data archived successfully', [
                'report_id' => $archiveReport->id,
                'old_phase' => $oldPhase->name,
                'new_phase' => $newPhase->name,
                'intake_id' => $activeIntake->id,
                'students_archived' => count($studentsData)
            ]);

        } catch (\Exception $e) {
            \Log::error('Error archiving phase data', [
                'error' => $e->getMessage(),
                'old_phase' => $oldPhase->name,
                'new_phase' => $newPhase->name
            ]);
            
            // Don't throw the exception - just log it and continue
            // This prevents the phase update from failing due to archiving issues
        }
    }

    /**
     * Get max scores from the active phase configuration
     * 
     * @return array
     */
    public function getMaxScores()
    {
        $activePhase = PhaseConfig::where('is_active', true)->first();
        
        if ($activePhase && $activePhase->max_scores) {
            return $activePhase->max_scores;
        }
        
        // Default values if no active phase or no max_scores
        return [
            'theory' => 40,
            'practical' => 30,
            '22cr' => 15,
            'pr' => 15
        ];
    }

    /**
     * Calculate grade based on percentage
     * 
     * @param float $percentage
     * @return string
     */
    private function calculateGrade($percentage)
    {
        $gradeScale = GradeScale::orderBy('min_score', 'desc')->get();
        
        foreach ($gradeScale as $grade) {
            if ($percentage >= $grade->min_score) {
                return $grade->grade;
            }
        }
        
        return 'F';
    }

    /**
     * Get comment based on percentage
     * 
     * @param float $percentage
     * @return string
     */
    private function getComment($percentage)
    {
        $gradeScale = GradeScale::orderBy('min_score', 'desc')->get();
        
        foreach ($gradeScale as $grade) {
            if ($percentage >= $grade->min_score) {
                return $grade->comment;
            }
        }
        
        return 'POOR';
    }

    /**
     * Show grade scale editing form
     * 
     * @return \Illuminate\View\View
     */
    public function editGradeScales()
    {
        $gradeScales = GradeScale::orderBy('min_score', 'desc')->get();
        return view('phase-config.edit-grades', compact('gradeScales'));
    }
    
    /**
     * Update grade scales
     * 
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateGradeScales(Request $request)
    {
        $grades = $request->grade;
        $minScores = $request->min_score;
        $comments = $request->comment;
        
        // Validate
        for ($i = 0; $i < count($grades); $i++) {
            $validator = Validator::make(
                [
                    'grade' => $grades[$i],
                    'min_score' => $minScores[$i],
                    'comment' => $comments[$i]
                ],
                [
                    'grade' => 'required|string|max:5',
                    'min_score' => 'required|numeric|min:0|max:100',
                    'comment' => 'required|string|max:255'
                ]
            );
            
            if ($validator->fails()) {
                return redirect()->route('phase-config.edit-grades')
                    ->withErrors($validator)
                    ->withInput();
            }
        }
        
        // Update grade scales
        GradeScale::truncate();
        
        for ($i = 0; $i < count($grades); $i++) {
            GradeScale::create([
                'grade' => $grades[$i],
                'min_score' => $minScores[$i],
                'comment' => $comments[$i]
            ]);
        }
        
        return redirect()->route('phase-config.index')
            ->with('success', 'Grade scales updated successfully.');
    }
    
    /**
     * Set a phase as active
     * 
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function setActive($id)
    {
        // Deactivate all phases
        PhaseConfig::where('is_active', true)->update(['is_active' => false]);
        
        // Activate the selected phase
        $phase = PhaseConfig::findOrFail($id);
        $phase->is_active = true;
        $phase->save();
        
        return redirect()->route('phase-config.index')
            ->with('success', "Phase '{$phase->phase_name}' set as active.");
    }
}