<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Bn;
use App\Models\Coy;
use App\Models\Pl;
use App\Models\Admission;
use App\Models\Pr;
use Illuminate\Support\Facades\DB;
use App\Models\Phase;

use Maatwebsite\Excel\Facades\Excel;

class PRController extends Controller
{
/**
 * Display a listing of the resource.
 *
 * @param  \Illuminate\Http\Request  $request
 * @return \Illuminate\Http\Response
 */
public function index(Request $request)
{
    // Get the number of admissions to display per page from the request, default is 25
    $perPage = $request->input('per_page', 25);

    // Define the query for admissions
    $query = Admission::query();

    // Apply filters based on request parameters
    if ($request->filled('bn')) {
        $query->where('bn', $request->bn);
    }

    if ($request->filled('coy')) {
        $query->where('coy', $request->coy);
    }

    if ($request->filled('pl')) {
        $query->where('pl', $request->pl);
    }

    if ($request->filled('army_number')) {
        $query->where('army_number', 'like', '%' . $request->army_number . '%');
    }

    // Check if this is an AJAX request
    if ($request->ajax()) {
        $data = $query->get();
        return response()->json(['data' => $data]);
    }

    // Fetch paginated admissions based on the perPage value (for non-AJAX requests)
    $admissions = $query->paginate($perPage);

    // Fetch the filter options for the dropdowns
    $bn = Bn::pluck('name');
    $coy = Coy::pluck('name');
    $pl = Pl::pluck('name');

    // Return the view with the required data
    return view('pr.index', compact('bn', 'coy', 'pl', 'admissions'));
}




    public function storeMarks(Request $request)
    {
        foreach ($request->input('baters') as $studentId => $baters) {
            $pr = Pr::find($studentId);
    
            // Update the student's marks
            $pr->baters = $baters;  // Save the rating or calculate the score
            $pr->items = $request->input('items')[$studentId];
            $pr->sum_of_rating = $request->input('sum_of_rating')[$studentId];
            $pr->no_of_rating = $request->input('no_of_rating')[$studentId];
            $pr->mean_rating = $request->input('mean_rating')[$studentId];
            $pr->score = $request->input('score')[$studentId];
            $pr->save();
        }
    
        return redirect()->route('pr.index')->with('success', 'Marks updated successfully!');
    }


/**
 * Filter admissions via AJAX
 *
 * @param  \Illuminate\Http\Request  $request
 * @return \Illuminate\Http\Response
 */
public function filter(Request $request)
{
    // Define the query for admissions
    $query = Admission::query();

    // Apply filters based on request parameters
    if ($request->filled('bn')) {
        $query->where('bn', $request->bn);
    }

    if ($request->filled('coy')) {
        $query->where('coy', $request->coy);
    }

    if ($request->filled('pl')) {
        $query->where('pl', $request->pl);
    }

    if ($request->filled('army_number')) {
        $query->where('army_number', 'like', '%' . $request->army_number . '%');
    }

    // Get all filtered results (no pagination for AJAX)
    $data = $query->get();

    return response()->json(['data' => $data]);
}

    public function filterPlatoon(Request $request)
{
    // Get the filter parameters from the request
    $bn = $request->input('bn');
    $coy = $request->input('coy');
    $pl = $request->input('pl');
    $army_number = $request->input('army_number');

    // Query the database to filter students based on the selected parameters
    $students = Student::where('bn', $bn)
                        ->where('coy', $coy)
                        ->where('pl', $pl)
                        ->where('army_number', 'LIKE', "%$army_number%")
                        ->get();

    // Return the results as JSON
    return response()->json([
        'data' => $students
    ]);
}

    




public function create($admissionId)
{
    // Fetch the selected student using admissionId
    $selectedStudent = Admission::findOrFail($admissionId); // Renamed to $selectedStudent

    // Fetch all students who have the same bn, coy, and pl
    $students = Admission::where('bn', $selectedStudent->bn)
        ->where('coy', $selectedStudent->coy)
        ->where('pl', $selectedStudent->pl)
        ->get();

    $phases = Phase::get();

    return view('pr.create', compact('students', 'admissionId', 'selectedStudent', 'phases')); // Pass $selectedStudent to the view
}

    
    


public function store(Request $request)
{
    try {
        DB::beginTransaction();
        
        // Validate the request
        $request->validate([
            'items' => 'required|array',
            'items.*' => 'required|integer|between:2,6',
            'admission_id' => 'required|exists:admissions,id',
            'phase_id' => 'required|exists:phases,id' // Add phase validation
        ]);
        
        // Get the rater (person giving the ratings)
        $rater = Admission::findOrFail($request->admission_id);
        
        // Get the selected phase
        $selectedPhase = Phase::findOrFail($request->phase_id);
        
        // Counter for special ratings
        $ratingCounts = [
            2 => 0,
            6 => 0
        ];
        
        // Check if any special ratings exceed limits before processing
        foreach ($request->items as $studentId => $rating) {
            if ($rating == 2) {
                $ratingCounts[2]++;
            } elseif ($rating == 6) {
                $ratingCounts[6]++;
            }
        }
        
        // Validate special ratings don't exceed limits
        if ($ratingCounts[2] > 1) {
            throw new \Exception("You can only assign a rating of 2 to one person");
        }
        
        if ($ratingCounts[6] > 1) {
            throw new \Exception("You can only assign a rating of 6 to one person");
        }
        
        // Process each rating
        foreach ($request->items as $studentId => $rating) {
            // Get the student being rated
            $student = Admission::findOrFail($studentId);
            
            // Create record data with single score column
            $prData = [
                'army_number' => $student->army_number,
                'name' => $student->name,
                'rank' => $student->rank,
                'score' => $rating,
                'phase' => $selectedPhase->name, // Use selected phase name
                'admission_id' => $rater->id
            ];
            
            // Try to find existing record for the same rater, student, and phase
            $existingRecord = PR::where('admission_id', $rater->id)
                              ->where('army_number', $student->army_number)
                              ->where('phase', $selectedPhase->name)
                              ->first();
            
            if ($existingRecord) {
                // Update existing record
                $existingRecord->update($prData);
            } else {
                // Create new record
                PR::create($prData);
            }
            

        }
        
        DB::commit();
        return redirect()->route('pr.index')
            ->with('success', 'Peer ratings saved successfully for ' . $selectedPhase->name);
        
    } catch (\Exception $e) {
        DB::rollback();
        \Log::error('PR Save Error: ' . $e->getMessage());
        return redirect()->route('pr.create', $request->admission_id)
            ->with('error', 'Error saving peer ratings: ' . $e->getMessage());
    }
}



    public function showPRList($admissionId) {
    

        try {
            // Get all PR records for the admission_id

          $phase  = Phase::where('is_active', 1)->get()->first();

            $prRecords = PR::where('admission_id', $admissionId)->where('phase',$phase->name)->get();

            
  
    
            // Calculate statistics for each student
            $prStats = $prRecords->map(function ($pr) use ($prRecords) {
                // Calculate sum of ratings (s2 through s6)
                $sumOfRatings = collect(range(2, 6))
                    ->map(function ($i) use ($pr) {
                        return $pr->{"s$i"} * $i; // Multiply the count by the rating value
                    })
                    ->sum();
    
                // Count number of raters (total of s2 through s6 values)
                $numberOfRaters = collect(range(2, 6))
                    ->map(function ($i) use ($pr) {
                        return $pr->{"s$i"};
                    })
                    ->sum();
    
                // Calculate mean (if there are raters)
                $meanRating = $numberOfRaters > 0 ? $sumOfRatings / $numberOfRaters : 0;
                $meanRatingRounded = round($meanRating, 2);
    
                // Calculate PR score based on the updated rating range (2-6)
                $score = 0;
                if ($meanRatingRounded > 0) {
                    $wholeNumber = floor($meanRatingRounded);
                    $decimalPart = $meanRatingRounded - $wholeNumber;
                    
                    // Get assigned weight based on the table (adjusted for 2-6 range)
                    $assignedWeights = [
                        2 => 10,
                        3 => 14,
                        4 => 18,
                        5 => 22,
                        6 => 26
                    ];
                    
                    if (isset($assignedWeights[$wholeNumber])) {
                        $baseWeight = $assignedWeights[$wholeNumber];
                        $additionalWeight = $decimalPart * 4; // Constant interval is 4
                        $totalWeight = $baseWeight + $additionalWeight;
                        
                        // Calculate final score (divide by 30 and multiply by 10%)
                        $score = ($totalWeight / 30) * 10;
                        $score = round($score, 1);
                    }
                }
    
                return [
                    'id' => $pr->id,
                    'army_number' => $pr->army_number,
                    'name' => $pr->name,
                    'rank' => $pr->rank,
                    'ratings' => [
                        's2' => $pr->s2,
                        's3' => $pr->s3,
                        's4' => $pr->s4,
                        's5' => $pr->s5,
                        's6' => $pr->s6,
                    ],
                    'sum_of_ratings' => $sumOfRatings,
                    'number_of_raters' => $numberOfRaters,
                    'mean_rating' => $meanRatingRounded,
                    'score' => $score,
                ];
            });
    
            return view('pr.list', [
                'admissionId' => $admissionId,
                'prStats' => $prStats,  // Added this to pass the calculated stats to the view
            ]);
        } catch (\Exception $e) {
            \Log::error('PR List Error: ' . $e->getMessage());
            return back()->with('error', 'Error loading PR list: ' . $e->getMessage());
        }
    }
    public function updateRating(Request $request)
    {
        try {
            // Validate the request
            $validator = \Validator::make($request->all(), [
                'id' => 'required',
                'current_score' => 'required|integer|between:2,6',
                'new_score' => 'required|integer|between:2,6',
                'admission_id' => 'required'
            ]);
    
            if ($validator->fails()) {
                return response()->json([
                    'error' => 'Validation failed',
                    'messages' => $validator->errors()
                ], 422);
            }
    
            DB::beginTransaction();
    
            // Find the PR record to update
            $pr = PR::findOrFail($request->id);
    
            // Validate special ratings (2 or 6)
            // if (in_array($request->new_score, [2, 6])) {
            //     // Check if another rating of 2 or 6 exists for the same admission
            //     $existingSpecialRating = PR::where('admission_id', $request->admission_id)
            //         ->where('id', '!=', $request->id)
            //         ->where('score', $request->new_score)
            //         ->exists();
    
            //     if ($existingSpecialRating) {
            //         throw new \Exception("You can only give one rating of {$request->new_score}");
            //     }
            // }
    
            // Update the score
            $pr->score = $request->new_score;
            $pr->save();
    
            DB::commit();
    
            return response()->json([
                'success' => true,
                'message' => 'Rating updated successfully'
            ]);
    
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('PR Update Error: ' . $e->getMessage());
            return response()->json([
                'error' => $e->getMessage()
            ], 400);
        }
    }
public function listRaters(Request $request)
{
    // Get sort parameters (with defaults)
    $sortField = $request->input('sort', 'army_number');
    $sortDirection = $request->input('direction', 'asc');

    // Valid sort fields
    $validSortFields = ['army_number', 'name', 'rank', 'bn', 'phase', 'total_score', 'mean_score', 'final_score'];

    // Build query
    $query = Admission::with(['receivedRatings.rater'])
        ->whereHas('receivedRatings');

    // Filters
    if ($request->filled('search')) {
        $search = $request->search;
        $query->where('army_number', 'LIKE', "%{$search}%");
    }
    if ($request->filled('bn')) {
        $query->where('bn', $request->bn);
    }
    if ($request->filled('coy')) {
        $query->where('coy', $request->coy);
    }
    if ($request->filled('pl')) {
        $query->where('pl', $request->pl);
    }
    if ($request->filled('phase')) {
        $query->where('phase', $request->phase);
    }

    // Sort base fields
    if (in_array($sortField, ['army_number', 'name', 'rank', 'bn', 'phase'])) {
        $query->orderBy($sortField, $sortDirection);
    }

    // For filters in UI
    $battalions = Admission::distinct()->pluck('bn')->sort();
    $companies = Admission::distinct()->pluck('coy')->sort();
    $platoons = Admission::distinct()->pluck('pl')->sort();
    $phases = Phase::get();

    // Fetch data
    $ratersData = $query->get()->map(function ($rater) {

        // Check if this group has PRs with marked_by
        $hasMarkedByInGroup = DB::table('prs')
            ->join('admissions', 'prs.admission_id', '=', 'admissions.id')
            ->whereNotNull('prs.marked_by')
            ->where('admissions.bn', $rater->bn)
            ->where('admissions.coy', $rater->coy)
            ->where('admissions.pl', $rater->pl)
            ->exists();

        // Determine total peers
        if ($hasMarkedByInGroup) {
            $totalPeers = Admission::where('bn', $rater->bn)
                ->where('coy', $rater->coy)
                ->where('pl', $rater->pl)
                ->where('is_decommissioned', 0)
                ->count();
        } else {
            $totalPeers = Admission::where('bn', $rater->bn)
                ->where('coy', $rater->coy)
                ->where('pl', $rater->pl)
                ->where('id', '!=', $rater->id)
                ->where('is_decommissioned', 0)
                ->count();
        }

        // Calculate total score and ratings
        $totalScore = 0;
        $ratingsCount = 0;

        foreach ($rater->receivedRatings as $rating) {
            $totalScore += $rating->score;
            $ratingsCount++;
        }

        $rater->total_peers = $totalPeers;
        $rater->total_score = $totalScore;
        $rater->ratings_count = $ratingsCount;

        // Default values
        $rater->mean_score = null;
        $rater->final_score = 0;

        if ($hasMarkedByInGroup) {
            // 🟢 If PRs have marked_by, skip mean and just scale the total score (e.g., 89.9 → 8.9)
            $rater->final_score = round($totalScore / 10, 1);
        } else {
            // 🔵 Peer-based mean and weighted scoring
            $rater->mean_score = ($ratingsCount > 0 && $totalPeers > 0)
                ? round($totalScore / $totalPeers, 2)
                : 0;

            $meanRating = $rater->mean_score;
            $score = 0;

            if ($meanRating > 0) {
                $wholeNumber = floor($meanRating);
                $decimalPart = $meanRating - $wholeNumber;

                $assignedWeights = [
                    1 => 6,
                    2 => 10,
                    3 => 14,
                    4 => 18,
                    5 => 22,
                    6 => 26,
                    7 => 30
                ];

                if (isset($assignedWeights[$wholeNumber])) {
                    $baseWeight = $assignedWeights[$wholeNumber];
                    $additionalWeight = $decimalPart * 4;
                    $totalWeight = $baseWeight + $additionalWeight;
                    $score = ($totalWeight / 30) * 10;
                    $score = round($score, 1);
                }
            }

            $rater->final_score = $score;
        }

        return $rater;
    });

    // Sort calculated fields
    if (in_array($sortField, ['total_score', 'mean_score', 'final_score'])) {
        $ratersData = $sortDirection === 'asc'
            ? $ratersData->sortBy($sortField)
            : $ratersData->sortByDesc($sortField);
    }

    // Return to view
    return view('pr.raters', compact(
        'ratersData', 'battalions', 'companies', 'platoons',
        'phases', 'sortField', 'sortDirection'
    ));
}


public function markPRs()
    {
        // Get filter parameters
        $filters = [
            'bn' => request('bn'),
            'coy' => request('coy'),
            'pl' => request('pl'),
            'sec' => request('sec'),
            'marked' => request('marked'),
        ];

        // Query admissions with optional filters
        $query = Admission::query();
        
        if (!empty($filters['bn'])) {
            $query->where('bn', $filters['bn']);
        }
        
        if (!empty($filters['coy'])) {
            $query->where('coy', $filters['coy']);
        }
        
        if (!empty($filters['pl'])) {
            $query->where('pl', $filters['pl']);
        }
        
        if (!empty($filters['sec'])) {
            $query->where('sec', $filters['sec']);
        }

        $admissions = $query->orderBy('army_number')->get();

        // Get existing PR records to check marked status
        $existingPRs = PR::whereIn('admission_id', $admissions->pluck('id'))->get()->keyBy('admission_id');

        // Get filter values
        $filterValues = $this->getFilterValues();

        return view('pr.mark', array_merge(compact('admissions', 'filters', 'existingPRs'), $filterValues));
    }


public function updatePRs(Request $request)
    {
        $scores = $request->input('scores', []);
        
        foreach ($scores as $admissionId => $scoreData) {
            if (isset($scoreData['score'])) {
                $pr = PR::where('admission_id', $admissionId)->first();
                
                if ($pr) {
                    // Update existing PR record
                    $pr->update([
                        'score' => $scoreData['score'],
                        'marked_by' => auth()->user()->name ?? 'System',
                        'updated_at' => now()
                    ]);
                } else {
                    // Create new PR record
                    $admission = Admission::find($admissionId);
                    if ($admission) {
                        PR::create([
                            'army_number' => $admission->army_number,
                            'name' => $admission->name,
                            'rank' => $admission->rank,
                            'score' => $scoreData['score'],
                            'admission_id' => $admissionId,
                            'phase' => 'Phase 3',
                            'marked_by' => auth()->user()->name ?? 'System',
                            'created_at' => now(),
                            'updated_at' => now()
                        ]);
                    }
                }
            }
        }

        return redirect()->route('prs.mark')->with('success', 'PR scores saved successfully!');
    }

public function importPRs(Request $request)
    {


        $request->validate([
            'import_file' => 'required|file|mimes:xlsx,xls,csv',
            'phase' => 'required|string'
        ]);

        try {
            $phase = $request->input('phase');
            $file = $request->file('import_file');
            
            $data = Excel::toArray([], $file);
            
            if (empty($data)) {
                return redirect()->back()->with('error', 'The uploaded file is empty.');
            }

            $rows = $data[0];
            
            // Find the header row (look for "S/NO" or "ARMY NO." in any row)
            $headerRowIndex = null;
            $expectedHeaders = ['S/NO', 'ARMY NO.', 'RANK', 'FULL NAMES', 'SEX', 'PR'];
            
            foreach ($rows as $index => $row) {
                $rowValues = array_map('strtoupper', array_map('trim', $row));
                if ((in_array('S/NO', $rowValues) || in_array('IS/NO', $rowValues)) && 
                    (in_array('ARMY NO.', $rowValues) || in_array('ARMY NO', $rowValues))) {
                    $headerRowIndex = $index;
                    break;
                }
            }

            if ($headerRowIndex === null) {
                return redirect()->back()->with('error', 'Could not find header row with columns: S/NO, ARMY NO., RANK, FULL NAMES, SEX, PR');
            }

            // Get the header and rows after it
            $header = array_map('strtoupper', array_map('trim', $rows[$headerRowIndex]));
            $dataRows = array_slice($rows, $headerRowIndex + 1);
            
            // Map column indexes - handle variations in header names
            $columnIndexes = [];
            $columnMapping = [
                'S/NO' => ['S/NO', 'IS/NO', 'SNO'],
                'ARMY NO.' => ['ARMY NO.', 'ARMY NO', 'ARMY_NO'],
                'RANK' => ['RANK', 'IRANK'],
                'FULL NAMES' => ['FULL NAMES', 'FULL NAMES', 'NAME'],
                'SEX' => ['SEX', 'ISEX'],
                'PR' => ['PR']
            ];

            foreach ($columnMapping as $standardName => $possibleNames) {
                $columnIndexes[$standardName] = false;
                foreach ($possibleNames as $possibleName) {
                    $index = array_search($possibleName, $header);
                    if ($index !== false) {
                        $columnIndexes[$standardName] = $index;
                        break;
                    }
                }
            }

            // Validate that all required columns are found
            $missingColumns = [];
            foreach (['ARMY NO.', 'PR'] as $requiredColumn) {
                if ($columnIndexes[$requiredColumn] === false) {
                    $missingColumns[] = $requiredColumn;
                }
            }

            if (!empty($missingColumns)) {
                return redirect()->back()->with('error', "Required columns not found: " . implode(', ', $missingColumns));
            }

            $importedCount = 0;
            $updatedCount = 0;
            $errors = [];

            DB::beginTransaction();

            foreach ($dataRows as $index => $row) {
                $lineNumber = $headerRowIndex + $index + 2; // +2 for header row and 1-based indexing
                
                try {
                    // Skip empty rows
                    if (empty(array_filter($row))) {
                        continue;
                    }

                    // Get values using column indexes
                    $armyNumber = trim($row[$columnIndexes['ARMY NO.']] ?? '');
                    $score = trim($row[$columnIndexes['PR']] ?? '');
                    $rank = $columnIndexes['RANK'] !== false ? trim($row[$columnIndexes['RANK']] ?? '') : '';
                    $fullName = $columnIndexes['FULL NAMES'] !== false ? trim($row[$columnIndexes['FULL NAMES']] ?? '') : '';
                    $sex = $columnIndexes['SEX'] !== false ? trim($row[$columnIndexes['SEX']] ?? '') : '';

                    // Validate required fields
                    if (empty($armyNumber)) {
                        $errors[] = "Row {$lineNumber}: Army number is required";
                        continue;
                    }

                    if (empty($score)) {
                        $errors[] = "Row {$lineNumber}: PR score is required";
                        continue;
                    }

                    // Validate score is numeric
                    if (!is_numeric($score)) {
                        $errors[] = "Row {$lineNumber}: PR score must be numeric (found: '{$score}')";
                        continue;
                    }

                    $score = floatval($score);

                    // Find admission by army number
                    $admission = Admission::where('army_number', $armyNumber)->first();

                    if (!$admission) {
                        $errors[] = "Row {$lineNumber}: Admission not found for army number: {$armyNumber}";
                        continue;
                    }

                    // Check if PR already exists for this admission
                    $existingPR = PR::where('admission_id', $admission->id)->first();

                    if ($existingPR) {
                        // Update existing record
                        $existingPR->update([
                            'score' => $score,
                            'phase' => $phase,
                            'marked_by' => auth()->user()->name ?? 'System (Import)',
                            'updated_at' => now()
                        ]);
                        $updatedCount++;
                    } else {
                        // Create new record
                        PR::create([
                            'army_number' => $admission->army_number,
                            'name' => $admission->name,
                            'rank' => $admission->rank,
                            'score' => $score,
                            'admission_id' => $admission->id,
                            'phase' => $phase,
                            'marked_by' => auth()->user()->name ?? 'System (Import)',
                            'created_at' => now(),
                            'updated_at' => now()
                        ]);
                        $importedCount++;
                    }

                } catch (\Exception $e) {
                    $errors[] = "Row {$lineNumber}: " . $e->getMessage();
                    continue;
                }
            }

            DB::commit();

            $message = "PR Import completed successfully!";
            $message .= " New records: {$importedCount}, Updated records: {$updatedCount}";
            
            if (!empty($errors)) {
                $message .= ". Errors: " . count($errors);
                session()->flash('import_errors', $errors);
            }

            return redirect()->route('prs.mark')->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'PR Import failed: ' . $e->getMessage());
        }
    }

    public function showPRImportForm()
    {
        $phases = [
            'Phase 1',
            'Phase 2', 
            'Phase 3',
            'Phase 4',
            'Phase 5'
        ];
        
        return view('pr.import', compact('phases'));
    }



public function getFilterValues()
    {
        return [
            'bns' => Admission::distinct()->whereNotNull('bn')->pluck('bn')->toArray(),
            'coys' => Admission::distinct()->whereNotNull('coy')->pluck('coy')->toArray(),
            'pls' => Admission::distinct()->whereNotNull('pl')->pluck('pl')->toArray(),
            'secs' => Admission::distinct()->whereNotNull('sec')->pluck('sec')->toArray(),
        ];
    }

    // Calculate remarks based on score
    public function calculateRemarks($score)
    {
        if (!$score) return '';

        $numericScore = floatval($score);
        
        if ($numericScore >= 97.2) return '9.72 - EXCELLENT';
        if ($numericScore >= 90) return '9.0+ - VERY GOOD';
        if ($numericScore >= 80) return '8.0+ - GOOD';
        if ($numericScore >= 70) return '7.0+ - AVERAGE';
        if ($numericScore >= 60) return '6.0+ - BELOW AVERAGE';
        return 'FAIL';
    }
  
}
