<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithCustomStartCell;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class StudentsExport implements FromCollection, WithHeadings, WithStyles, WithTitle, WithCustomStartCell, WithEvents
{
    protected $studentsData;
    protected $metadata;

    public function __construct($studentsData, $metadata = [])
    {
        $this->studentsData = collect($studentsData);
        $this->metadata = $metadata;
    }

    /**
     * Return the collection of student data
     */
    public function collection()
    {
        return $this->studentsData->map(function($student, $index) {
            // 🔍 Debug log for each student
            \Log::info('Excel Export Student Data:', [
                'army_number' => $student['army_number'] ?? 'N/A',
                'cr_22_raw' => $student['cr_22'] ?? 'MISSING',
                'pr_raw' => $student['pr'] ?? 'MISSING'
            ]);

            return [
                'no' => $index + 1,
                'admission_number' => $student['admission_number'] ?? '',
                'army_number' => $student['army_number'] ?? '',
                'rank' => $student['rank'] ?? '',
                'name' => $student['name'] ?? '',
                'school' => $student['school'] ?? '',
                'course' => $student['course'] ?? '',
                'theory' => $student['theory'] ?? '0.00',
                'practical' => $student['practical'] ?? '0.00',
                // ✅ FIXED: Direct assignment without condition
                'cr_22' => $student['cr_22'] ?? '0.00',
                'pr' => $student['pr'] ?? '0.00',
                'total' => $student['total'] ?? '0.00',
                'grade' => $student['grade'] ?? '',
                'comment' => $student['comment'] ?? '',
                'remarks' => $student['remarks'] ?? ''
            ];
        });
    }

    /**
     * Define the headings for the Excel sheet
     */
    public function headings(): array
    {
        return [
            'No.',
            'Admission No.',
            'Army Number',
            'Rank',
            'Name',
            'School',
            'Course',
            'Theory',
            'Practical',
            '22 CR',
            'PR',
            'Total',
            'Grade',
            'Comment',
            'Remarks'
        ];
    }

    /**
     * Set the starting cell for data (to leave room for metadata)
     */
    public function startCell(): string
    {
        return 'A8'; // Start data from row 8
    }

    /**
     * Define the title of the sheet
     */
    public function title(): string
    {
        return 'GPV Report';
    }

    /**
     * Apply styles to the worksheet
     */
    public function styles(Worksheet $sheet)
    {
        // Title row (A1)
        $sheet->mergeCells('A1:O1');
        $sheet->setCellValue('A1', $this->metadata['title'] ?? 'GPV Report');
        $sheet->getStyle('A1')->applyFromArray([
            'font' => [
                'bold' => true,
                'size' => 16,
                'color' => ['rgb' => '000000']
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER
            ]
        ]);

        // Header row styling (row 8)
        $sheet->getStyle('A8:O8')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4472C4']
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => '000000']
                ]
            ]
        ]);

        // Auto-size columns
        foreach (range('A', 'O') as $column) {
            $sheet->getColumnDimension($column)->setAutoSize(true);
        }

        // Set row height for header
        $sheet->getRowDimension(8)->setRowHeight(25);

        return [];
    }

    /**
     * Register events for additional customization
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                
                // Add metadata information (rows 3-6)
                $metadataRow = 3;
                $metadata = [
                    ['School:', $this->metadata['school'] ?? 'N/A', '', 'Intake:', $this->metadata['intake'] ?? 'N/A'],
                    ['Course:', $this->metadata['course'] ?? 'N/A', '', 'Phase:', $this->metadata['phase'] ?? 'N/A'],
                    ['Battalion:', $this->metadata['bn'] ?? 'All', '', 'Company:', $this->metadata['coy'] ?? 'All'],
                    ['Platoon:', $this->metadata['pl'] ?? 'All', '', 'Grade Filter:', $this->metadata['grade'] ?? 'All'],
                    ['Date Generated:', $this->metadata['date_generated'] ?? date('F d, Y'), '', 'Total Students:', $this->metadata['total_students'] ?? 0]
                ];

                foreach ($metadata as $index => $row) {
                    $currentRow = $metadataRow + $index;
                    $sheet->setCellValue('A' . $currentRow, $row[0]);
                    $sheet->setCellValue('B' . $currentRow, $row[1]);
                    $sheet->setCellValue('D' . $currentRow, $row[3]);
                    $sheet->setCellValue('E' . $currentRow, $row[4]);
                    
                    // Style metadata labels (bold)
                    $sheet->getStyle('A' . $currentRow)->getFont()->setBold(true);
                    $sheet->getStyle('D' . $currentRow)->getFont()->setBold(true);
                }

                // Apply borders to data table
                $lastRow = $sheet->getHighestRow();
                $sheet->getStyle('A8:O' . $lastRow)->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_THIN,
                            'color' => ['rgb' => '000000']
                        ]
                    ],
                    'alignment' => [
                        'vertical' => Alignment::VERTICAL_CENTER
                    ]
                ]);

                // Center align specific columns
                $sheet->getStyle('A9:A' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                $sheet->getStyle('H9:L' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                $sheet->getStyle('M9:O' . $lastRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

                // Highlight PASSED/FAILED in remarks column
                for ($row = 9; $row <= $lastRow; $row++) {
                    $remarksValue = $sheet->getCell('O' . $row)->getValue();
                    if ($remarksValue === 'PASSED') {
                        $sheet->getStyle('O' . $row)->applyFromArray([
                            'font' => ['color' => ['rgb' => '008000'], 'bold' => true]
                        ]);
                    } elseif ($remarksValue === 'FAILED') {
                        $sheet->getStyle('O' . $row)->applyFromArray([
                            'font' => ['color' => ['rgb' => 'FF0000'], 'bold' => true]
                        ]);
                    }
                }

                // 🔍 Debug: Log the actual cell values
                \Log::info('Excel Cell Values (Row 9):', [
                    'cr_22_cell' => $sheet->getCell('J9')->getValue(),
                    'pr_cell' => $sheet->getCell('K9')->getValue()
                ]);

                // Add footer with generation info
                $footerRow = $lastRow + 2;
                $sheet->setCellValue('A' . $footerRow, 'Report generated by STRART System');
                $sheet->getStyle('A' . $footerRow)->applyFromArray([
                    'font' => ['italic' => true, 'size' => 9]
                ]);
            }
        ];
    }
}