<?php

namespace App\Exports;

use App\Models\Assessment;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class AssessmentsExport implements FromCollection, WithHeadings, WithStyles, ShouldAutoSize
{
    protected $filters;

    public function __construct(array $filters)
    {
        $this->filters = $filters;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $query = Assessment::with('admission');



        // Apply filters
        if (!empty($this->filters['assessment_type'])) {
            $query->where('assessment_type', $this->filters['assessment_type']);
        }

        if (!empty($this->filters['month'])) {
            $query->where('month', $this->filters['month']);
        }

        if (!empty($this->filters['assessment_group'])) {
            $query->where('assessment_group', $this->filters['assessment_group']);
        }

        if (!empty($this->filters['type'])) {
            $query->where('type', $this->filters['type']);
        }

        if (!empty($this->filters['week'])) {
            $query->where('week', $this->filters['week']);
        }

        if (!empty($this->filters['subject_id'])) {
            $query->where('subject_id', $this->filters['subject_id']);
        }

        if (!empty($this->filters['exercises'])) {
            $query->where('exercise', $this->filters['exercises']);
        }

        // Organization filters
        if (!empty($this->filters['bn'])) {
            $query->where('bn', $this->filters['bn']);
        }

        if (!empty($this->filters['coy'])) {
            $query->where('coy', $this->filters['coy']);
        }

        if (!empty($this->filters['pl'])) {
            $query->where('pl', $this->filters['pl']);
        }

        if (!empty($this->filters['sec'])) {
            $query->where('sec', $this->filters['sec']);
        }

        // Get results
        $assessments = $query->get()->groupBy('admission_id');

        // Format for export
        $exportData = collect();

        foreach ($assessments as $studentData) {
            $firstAssessment = $studentData->first();
            $student = [
                'army_number' => $firstAssessment->admission->army_number,
                'rank' => $firstAssessment->admission->rank,
                'name' => $firstAssessment->admission->name,
            ];

            // Get assessment marks based on type
            $assessmentMarks = $studentData->groupBy(function($assessment) {
                if ($assessment->type === 'Monthly') {
                    return 'monthly' . preg_replace('/[^0-9]/', '', $assessment->month, -1);
                } elseif ($assessment->type === 'End of Phase Exam') {
                    return 'eope';
                } else {
                    return preg_replace('/[^0-9]/', '', $assessment->week, -1);
                }
            });

            // Add marks based on assessment type
            if (!empty($this->filters['type']) && $this->filters['type'] == 'Monthly') {
                $monthlyColumns = $studentData->filter(function($assessment) {
                    return $assessment->type === 'Monthly';
                })->pluck('month')->unique()->sort();

                foreach ($monthlyColumns as $month) {
                    $monthKey = 'monthly' . substr($month, -1);
                    $assessment = isset($assessmentMarks[$monthKey]) ? $assessmentMarks[$monthKey]->first() : null;
                    $student['Monthly ' . substr($month, -1)] = $assessment ? $assessment->marks . '/' . $assessment->weight : 'N/A';
                }
            } elseif (!empty($this->filters['type']) && $this->filters['type'] == 'End of Phase Exam') {
                $assessment = isset($assessmentMarks['eope']) ? $assessmentMarks['eope']->first() : null;
                $student['EOPE'] = $assessment ? $assessment->marks . '/' . $assessment->weight : 'N/A';
            } else {
                // Weekly assessments
                for ($week = 1; $week <= 10; $week++) {
                    $assessment = isset($assessmentMarks[$week]) ? $assessmentMarks[$week]->first() : null;
                    $student['Week ' . $week] = $assessment ? $assessment->marks . '/' . $assessment->weight : 'N/A';
                }
            }

            $exportData->push($student);
        }

        return $exportData;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        $headings = [
            'Army Number',
            'Rank',
            'Cadet Name',
        ];

        // Add dynamic headings based on assessment type
        if (!empty($this->filters['type'])) {
            if ($this->filters['type'] == 'Monthly') {
                // Get assessments to determine monthly columns
                $assessments = $this->getFilteredAssessments();
                $monthlyColumns = $assessments->first()?->filter(function($assessment) {
                    return $assessment->type === 'Monthly';
                })->pluck('month')->unique()->sort() ?? collect([]);

                foreach ($monthlyColumns as $month) {
                    $headings[] = 'Monthly ' . substr($month, -1);
                }
            } elseif ($this->filters['type'] == 'End of Phase Exam') {
                $headings[] = 'EOPE';
            } else {
                // Weekly
                for ($i = 1; $i <= 10; $i++) {
                    $headings[] = 'Week ' . $i;
                }
            }
        }

        return $headings;
    }

    /**
     * Apply styles to Excel sheet
     */
    public function styles(Worksheet $sheet)
    {
        // Style the header row
        $sheet->getStyle('A1:Z1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4CAF50'], // Green color matching your table header
            ],
        ]);

        // Style the monthly columns if applicable
        if (!empty($this->filters['type']) && $this->filters['type'] == 'Monthly') {
            $lastColumn = $sheet->getHighestColumn();
            $columnIndex = 4; // D column (after Army Number, Rank, Name)

            while ($columnIndex <= \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($lastColumn)) {
                $column = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($columnIndex);
                $sheet->getStyle($column . '1')->applyFromArray([
                    'fill' => [
                        'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                        'startColor' => ['rgb' => '2196F3'], // Blue color for Monthly columns
                    ],
                ]);
                $columnIndex++;
            }
        } elseif (!empty($this->filters['type']) && $this->filters['type'] == 'End of Phase Exam') {
            $sheet->getStyle('D1')->applyFromArray([
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '9C27B0'], // Purple color for EOPE
                ],
            ]);
        }

        return $sheet;
    }

    /**
     * Get filtered assessments
     */
    private function getFilteredAssessments()
    {
        $query = Assessment::with('admission');

        foreach ($this->filters as $key => $value) {
            if (empty($value)) continue;

            if (in_array($key, ['assessment_type', 'assessment_group', 'type', 'week', 'subject_id', 'exercises', 'bn', 'coy', 'pl', 'sec'])) {
                $query->where($key, $value);
            }
        }

        return $query->get()->groupBy('admission_id');
    }
}
